#include <iostream>
#include <QtCore>
#include <QtNetwork>
#include "echoserver.h"

#define STR(s) ((QTextCodec::codecForLocale()->fromUnicode(s)).constData())

using namespace std;

//----------------------------------------

EchoThread::EchoThread(int socketDescriptor, QObject *parent)
          : QThread(parent), m_socketDescriptor(socketDescriptor) {}
          
void EchoThread::run(){
    m_client = new QTcpSocket();
    if (!m_client->setSocketDescriptor(m_socketDescriptor)) {
        emit error(m_client->error());
        return;
    }
    connect(m_client, SIGNAL(readyRead()), this,  SLOT(onRead()), 
            Qt::DirectConnection);
    connect(m_client, SIGNAL(error(QAbstractSocket::SocketError)),
            this, SIGNAL(error(QAbstractSocket::SocketError)), 
            Qt::DirectConnection);
    connect(this, SIGNAL(error(QAbstractSocket::SocketError)),
            this, SLOT(displayError(QAbstractSocket::SocketError)), 
            Qt::DirectConnection);
    connect(m_client, SIGNAL(disconnected()), this, SLOT(onDisconnect()), 
            Qt::DirectConnection);
    m_client->write(QTextCodec::codecForLocale()->fromUnicode(
                         QObject::tr("Hello! !") + 0x0d ) );
    m_addr = m_client->peerAddress();
    cout << STR(QObject::tr(" %1:      %2\n")
                         .arg(m_socketDescriptor).arg( m_addr.toString() ));
    exec();
}

void EchoThread::onRead() {
    QByteArray block = m_client->readAll();
    QString s = QTextCodec::codecForTr()->toUnicode(block.data());
    cout << STR(QObject::tr(" %1:      %2: %3\n")
                        .arg(m_socketDescriptor).arg(m_addr.toString()).arg(s) );
    m_client->write(QTextCodec::codecForLocale()->fromUnicode(s));
}

void EchoThread::onDisconnect() {
    m_client->deleteLater();
    cout << STR(QObject::tr("\n %1 \n").arg(m_socketDescriptor));
    quit();
}

void EchoThread::displayError(QAbstractSocket::SocketError error){
    if (error == QAbstractSocket::RemoteHostClosedError)
        cout << STR(QObject::tr("\n %1:  .")
                            .arg(m_socketDescriptor));
    else
        cout << STR(QObject::tr("\n: %1\n").arg(m_client->errorString()));
}

//----------------------------------------

EchoServer::EchoServer(QObject *parent) 
          : QTcpServer(parent) {}

bool EchoServer::start() {
    QHostAddress addr;
    if ( !listen(QHostAddress::Any, 5678) ) {
        cout << STR(QObject::tr("   : %1\n")
                   .arg( errorString() ) );
        close();
        return false;
    }
    cout << STR(QObject::tr(" \n"));
    return true;
}

void EchoServer::incomingConnection(int socketDescriptor) {
    EchoThread *thread = new EchoThread(socketDescriptor, this);
    connect(thread, SIGNAL(finished()), this, SLOT(removeThread()), 
            Qt::DirectConnection);
    m_threads.append(thread);
    thread->start();
}

void EchoServer::removeThread(){
    EchoThread *thread = qobject_cast<EchoThread*>(sender());
    m_threads.removeAt(m_threads.indexOf(thread));
    thread->deleteLater();
}

//===============================================
int main(int argc, char **argv) {
    QCoreApplication app(argc, argv);
    QTextCodec *incodec = QTextCodec::codecForName("CP1251");
    
    #ifdef Q_WS_WIN
    QTextCodec *outcodec = QTextCodec::codecForName("CP866");
    QTextCodec::setCodecForLocale(outcodec);
    #endif

    QTextCodec::setCodecForTr(incodec);
    
    qRegisterMetaType<QAbstractSocket::SocketError>("QAbstractSocket::SocketError");

    EchoServer server;
    server.start();
    return app.exec();
}
