#include "xmlstreamreaderhelper.h"
#include <QtCore/QtGlobal>
#include <QtCore/QtDebug>
#include <QtCore/QCoreApplication>

QXmlStreamReader *AbstractTagImplementator::m_xml = 0;

AbstractTagImplementator::AbstractTagImplementator(const QString &name, 
        AbstractTagImplementator *parent) : m_tagName(name)
{
    setParent(parent);
}

void AbstractTagImplementator::setParent(AbstractTagImplementator *parent)
{
    if (parent && !m_tagName.isNull())
        parent->setProcessor(m_tagName, this);
}

bool AbstractTagImplementator::beforeEvent()
{
    return false;
}

bool AbstractTagImplementator::parse()
{
    if (!xml()) {
        qCritical() << "No valid QXmlStreamReader pointer found." << __LINE__ << __FILE__;
        return true;
    }

    if (beforeEvent())
        return false;
    cycle();
    afterEvent();
    return xml()->error() ? true : false;
}

void AbstractTagImplementator::cycle()
{
    do{
        while (!xml()->atEnd()){
            token = xml()->readNext();

            if (event())
                return;
        }
    } while (!isCritical() && token != QXmlStreamReader::EndDocument);
}

bool AbstractTagImplementator::event()
{
    return processTag();
}

bool AbstractTagImplementator::processTag()
{
    if (token == QXmlStreamReader::StartElement) {
        if (!processors.isEmpty()) {
            AbstractTagImplementator *tag = processor(xml()->name().toString());
            if (tag && tag->parse())
                return true;
        }
    } else if (token == QXmlStreamReader::EndElement && xml()->name() == name()) {
        return true;
    }
    return false;
}

void AbstractTagImplementator::afterEvent()
{
}

bool AbstractTagImplementator::isCritical()
{
    if (xml()->hasError()) {
        if (xml()->error() == QXmlStreamReader::PrematureEndOfDocumentError)
            return appendMoreData();
        else
            qDebug() << "Error: " << xml()->errorString();
        return true;
    }
    return false;
}

bool AbstractTagImplementator::appendMoreData()
{
    //  error or quit request
    return AbstractTagImplementator::eventLoop()->exec() ? true : false;
}

QEventLoop * AbstractTagImplementator::eventLoop()
{
    static QSharedPointer<QEventLoop> loop = QSharedPointer<QEventLoop>(new QEventLoop);
    return loop.data();
}

void AbstractTagImplementator::setProcessor(const QString &tagName, AbstractTagImplementator *processor)
{
    processors[tagName] = processor;
}

AbstractTagImplementator * AbstractTagImplementator::processor(const QString &tagName) const
{
    return processors.value(tagName);
}

AbstractTagImplementator::~AbstractTagImplementator()
{
}

AbstractTagProcessor::AbstractTagProcessor(
        const QString &name, AbstractTagImplementator *parent) :
        AbstractTagImplementator(name, parent)
{
}

bool AbstractTagProcessor::skipSubTree()
{
    return m_skipSubTree.parse();
}

bool AbstractTagProcessor::moveToNextTag()
{
    return m_moveToNextTag.parse();
}

bool AbstractTagProcessor::moveToTag(const QString &tagName)
{
    return m_moveToTag.moveTo(tagName);
}

bool AbstractTagProcessor::skipTags(quint32 count)
{
    return m_skipTags.skip(count);
}

bool SkipSubTree::beforeEvent()
{
    tagName = xml()->name().toString();
    return false;
}

bool SkipSubTree::event()
{
    if (token == QXmlStreamReader::EndElement && tagName == xml()->name())
        return true;
    return false;
}

bool MoveToNextTag::event()
{
    if (token == QXmlStreamReader::StartElement)
        return true;
    return false;
}

bool MoveToTag::event()
{
    return token == QXmlStreamReader::StartElement && xml()->name() == m_tagName ? true : false;
}

bool MoveToTag::moveTo(const QString &tagName)
{
    m_tagName = tagName;
    return parse();
}

bool SkipTags::skip(quint32 count)
{
    m_count = count;
    return parse();
}

bool SkipTags::event()
{
    if (m_count && token == QXmlStreamReader::StartElement) {
        --m_count;
        return false;
    }
    return true;
}

