#include "port.h"
#include <QtCore/QEvent>
#include <QtCore/QDebug>

Port::Port(QObject *parent)
    : QWinEventNotifier(parent)
    , m_hd(INVALID_HANDLE_VALUE)
    , m_setMask(0)
    , m_currMask(0)
{
    int size = sizeof(OVERLAPPED);
    memset(&m_or, 0, size);
    memset(&m_ow, 0, size);
    memset(&m_oe, 0, size);

    m_or.hEvent = ::CreateEventA(0, false, false, 0);
    m_ow.hEvent = ::CreateEventA(0, false, false, 0);
    m_oe.hEvent = ::CreateEventA(0, false, false, 0);
    setHandle(m_oe.hEvent);

    m_setMask = EV_BREAK | EV_CTS | EV_DSR | EV_ERR | EV_RING | EV_RLSD | EV_RXCHAR | EV_RXFLAG | EV_TXEMPTY;
    setEnabled(true);
}

Port::~Port()
{
    setEnabled(false);
    ::CloseHandle(m_or.hEvent);
    ::CloseHandle(m_ow.hEvent);
    ::CloseHandle(m_oe.hEvent);
}

void Port::control()
{
    if (isOpen()) {
        setEnabled(false);
        ::CancelIo(m_hd);
        ::CloseHandle(m_hd);
        m_hd = INVALID_HANDLE_VALUE;
    } else {
        m_hd = ::CreateFileA("\\\\.\\COM3", GENERIC_READ | GENERIC_WRITE,
                             0, 0, OPEN_EXISTING, FILE_FLAG_OVERLAPPED, 0);

        if (isOpen()) {
            DCB dcb;

            if (::GetCommState(m_hd, &dcb)) {
                dcb.BaudRate = CBR_19200;
                dcb.ByteSize = 8;
                dcb.DCBlength = sizeof(DCB);
                dcb.fAbortOnError = false;
                dcb.fBinary = true;
                dcb.fDsrSensitivity = false;
                dcb.fDtrControl = false;
                dcb.fOutxCtsFlow = false;
                dcb.fOutxDsrFlow = false;
                dcb.fParity = false;
                dcb.fRtsControl = false;
                dcb.Parity = NOPARITY;
                dcb.StopBits = ONESTOPBIT;

                if (::SetCommState(m_hd, &dcb)) {

                    COMMTIMEOUTS ct;
                    if (::GetCommTimeouts(m_hd, &ct)) {
                        ct.ReadIntervalTimeout = MAXWORD;
                        ct.ReadTotalTimeoutConstant = 0;
                        ct.ReadTotalTimeoutMultiplier = 0;
                        ct.WriteTotalTimeoutConstant = 0;
                        ct.WriteTotalTimeoutMultiplier = 0;

                        if (::SetCommTimeouts(m_hd, &ct)) {
                            ::SetCommMask(m_hd, m_setMask);
                            ::WaitCommEvent(m_hd, &m_currMask, &m_oe);
                            setEnabled(true);
                            return;
                        }
                    }
                }
            }
            control();
        }
    }
}

static bool clear_overlapped(OVERLAPPED *o)
{
    o->Internal = o->InternalHigh = o->Offset = o->OffsetHigh = 0;
    return (0 != o->hEvent);
}

void Port::send(const QByteArray &data)
{
    if (!clear_overlapped(&m_ow))
        return;

    DWORD writeBytes = 0;
    if (::WriteFile(m_hd, data.constData(), data.count(), &writeBytes, &m_ow)) {
    } else {
        if (ERROR_IO_PENDING == ::GetLastError()) {
            switch (::WaitForSingleObject(m_ow.hEvent, 5000)) {
            case WAIT_OBJECT_0:
                if (::GetOverlappedResult(m_hd, &m_ow, &writeBytes, false)) {
                }
                break;
            default: ;
            }
        }
    }
}

int Port::recv(char *data, int len)
{
    clear_overlapped(&m_or);

    DWORD readBytes = 0;
    bool sucessResult = false;

    if (::ReadFile(m_hd, data, len, &readBytes, &m_or))
        sucessResult = true;
    else {
        if (::GetLastError() == ERROR_IO_PENDING) {
            // Instead of an infinite wait I/O (not looped), we expect, for example 5 seconds.
            // Although, maybe there is a better solution.
            switch (::WaitForSingleObject(m_or.hEvent, 5000)) {
            case WAIT_OBJECT_0: {
                if (::GetOverlappedResult(m_hd, &m_or, &readBytes, false))
                    sucessResult = true;
            }
            break;
            default: ;
            }
        }
    }

    if(!sucessResult) {
        readBytes = -1;
    }
    return int(readBytes);
}

int Port::avail() const
{
    DWORD err;
    COMSTAT cs;
    if (::ClearCommError(m_hd, &err, &cs) == 0)
        return -1;
    return int(cs.cbInQue);
}

bool Port::event(QEvent *e)
{
    bool ret = false;
    if (e->type() == QEvent::WinEventAct) {

        if (EV_RXCHAR & m_currMask & m_setMask) {
            //qDebug("Event: EV_RXCHAR\n");
            emit readyRead();
            ret = true;
        }
        if (EV_TXEMPTY & m_currMask & m_setMask) {
            //qDebug("Event: EV_TXEMPTY\n");
            ret = true;
        }

        if (!ret)  {
            //qDebug("Other event\n");
            ret = true;
        }
    } else
        ret = QWinEventNotifier::event(e);

    ::WaitCommEvent(m_hd, &m_currMask, &m_oe);
    return ret;
}
