#include <QHostAddress>
#include <QTimerEvent>

#include "sockets.h"

static const int RateControlTimerDelay = 2000;

/// Constructs an rate controlled socket and starts the rate control timer.
KTcpSocket::KTcpSocket(QObject *parent)
    : QTcpSocket(parent)
{
    memset(uploadSpeedData, 0, sizeof(uploadSpeedData));
    memset(downloadSpeedData, 0, sizeof(downloadSpeedData));

    transferSpeedTimer = startTimer(RateControlTimerDelay);

    connect(this, SIGNAL(readyRead()), this, SIGNAL(readyToTransfer()));
    connect(this, SIGNAL(connected()), this, SIGNAL(readyToTransfer()));

    connect(&socket, SIGNAL(connected()),
            this, SIGNAL(connected()));
    connect(&socket, SIGNAL(readyRead()),
            this, SIGNAL(readyRead()));
    connect(&socket, SIGNAL(disconnected()),
            this, SIGNAL(disconnected()));
    connect(&socket, SIGNAL(error(QAbstractSocket::SocketError)),
            this, SIGNAL(error(QAbstractSocket::SocketError)));
    connect(&socket, SIGNAL(bytesWritten(qint64)),
            this, SIGNAL(bytesWritten(qint64)));
}

/// Attempts to write 'bytes' bytes to the socket from the buffer.
/// This is used by RateController, which precisely controls how much each client can write.
qint64 KTcpSocket::writeToSocket(qint64 bytes)
{
    qint64 totalWritten = 0;
    do {
        qint64 written = socket.write(outgoingBuffer.constData(),
                                      qMin<qint64>(bytes - totalWritten, outgoingBuffer.size()));
        if (written <= 0)
            return totalWritten ? totalWritten : written;

        totalWritten += written;
        uploadSpeedData[0] += written;
        outgoingBuffer.remove(0, written);
    } while (totalWritten < bytes && !outgoingBuffer.isEmpty());

    return totalWritten;
}

/// Attempts to read at most 'bytes' bytes from the socket.
qint64 KTcpSocket::readFromSocket(qint64 bytes)
{
    char buffer[1024];
    qint64 totalRead = 0;
    do {
        qint64 bytesRead = socket.read(buffer, qMin<qint64>(sizeof(buffer), bytes - totalRead));
        if (bytesRead <= 0)
            break;
        qint64 oldSize = incomingBuffer.size();
        incomingBuffer.resize(oldSize + bytesRead);
        memcpy(incomingBuffer.data() + oldSize, buffer, bytesRead);

        totalRead += bytesRead;
    } while (totalRead < bytes);

    if (totalRead > 0) {
        downloadSpeedData[0] += totalRead;
        emit bytesReceived(totalRead);
		emit readyRead();
    }
    return totalRead;
}

/// Returns the average number of bytes per second this client is downloading.
qint64 KTcpSocket::downloadSpeed() const
{
    qint64 sum = 0;
    for (unsigned int i = 0; i < sizeof(downloadSpeedData) / sizeof(qint64); ++i)
        sum += downloadSpeedData[i];
    return sum / (8 * 2);
}

/// Returns the average number of bytes per second this client is uploading.
qint64 KTcpSocket::uploadSpeed() const
{
    qint64 sum = 0;
    for (unsigned int i = 0; i < sizeof(uploadSpeedData) / sizeof(qint64); ++i)
        sum += uploadSpeedData[i];
    return sum / (8 * 2);
}

void KTcpSocket::setReadBufferSize(int size)
{
    socket.setReadBufferSize(size);
}

bool KTcpSocket::canTransferMore() const
{
    return bytesAvailable() > 0 || socket.bytesAvailable() > 0 
			|| !outgoingBuffer.isEmpty();
}

void KTcpSocket::connectToHostImplementation(const QString &hostName, quint16 port, OpenMode openMode)
{
    setOpenMode(openMode);
    socket.connectToHost(hostName, port, openMode);
}

void KTcpSocket::diconnectFromHostImplementation()
{
    socket.disconnectFromHost();
}

void KTcpSocket::timerEvent(QTimerEvent *event)
{
    if (event->timerId() == transferSpeedTimer) {
        // Rotate the upload / download records.
        for (int i = 6; i >= 0; --i) {
            uploadSpeedData[i + 1] = uploadSpeedData[i];
            downloadSpeedData[i + 1] = downloadSpeedData[i];
        }
        uploadSpeedData[0] = 0;
        downloadSpeedData[0] = 0;
    }
    QTcpSocket::timerEvent(event);
}

qint64 KTcpSocket::readData(char *data, qint64 size)
{
    int n = qMin<int>(size, incomingBuffer.size());
    memcpy(data, incomingBuffer.constData(), n);
    incomingBuffer.remove(0, n);
    return n;
}

qint64 KTcpSocket::readLineData(char *data, qint64 maxlen)
{
    return QIODevice::readLineData(data, maxlen);
}

qint64 KTcpSocket::writeData(const char *data, qint64 size)
{
    int oldSize = outgoingBuffer.size();
    outgoingBuffer.resize(oldSize + size);
    memcpy(outgoingBuffer.data() + oldSize, data, size);
    emit readyToTransfer();
    return size;
}
