/*
* This file is part of QSerialDevice, an open-source cross-platform library
* Copyright (C) 2009  Denis Shienkov
*
* This library is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*
* Contact Denis Shienkov:
*          e-mail: <scapig2@yandex.ru>
*             ICQ: 321789831
*/


/*! \class AbstractSerial

    \brief \~russian Класс AbstractSerial предоставляет минимальный и достаточный интерфейс для работы с последовательными устройствами.

    \~russian Класс для работы с последовательными устройствами (портами).
    Этот класс является кросс - платформенной "низкоуровневой" библиотекой, которая основана на Qt4 и,
    соответственно, может использоваться для создания приложений Qt4.\n

    Реализация структуры этого класса во многом аналогична реализации QAbstractSocket из поставки Qt4 и
    соответствует идеологии открытых/закрытых классов (pimpl).\n

    Работа этого класса с последовательным устройством осуществляется в асинхронном режиме.
    Здесь под асинхронным режимом понимается тот факт, что само последовательное устройство открывается
    в неблокирующем режиме (но в итоге методы класса всё-равно ожидают завершения асинхронных операций ввода/вывода!).
    Т.е. при больших объемах передаваемых через устройство данных при использовании GUI возможны
    замирания GUI во время обработки методов чтения/записи данных, поэтому рекомендуется создавать
    объект класса в другом потоке.\n

    При использовании в различных операционных системах - используются их родные системные API:
    - в ОС Windows семейств NT/2K/XP и т.д. используется Win32 API.
    - в POSIX подобных ОС используются системные вызовы POSIX.

    Начало работы с классом необходимо начинать с создания экземпляра объекта AbstractSerial.\n
    Пример:
    \code
        ...
        AbstractSerial *serialDevice = new AbstractSerial(this);
        ...
    \endcode

    Также по умолчанию объекту присвоится первое имя реально существующего в системе устройства.

    \verbatim
        Операционная система:          Имя устройства по умолчанию:
        Windows                        "COM1"
        IRIX                           "/dev/ttyf1"
        HPUX                           "/dev/tty1p0"
        SOLARIS                        "/dev/ttya"
        FREEBSD                        "/dev/ttyd1"
        LINUX                          "/dev/ttyS0"
        <default>                      "/dev/ttyS0"
    \endverbatim

    Далее необходимо устройству(объекту) присвоить реально существующее имя в системе:
    - void AbstractSerial::setDeviceName(const QString &deviceName).

    Для открытия последовательного устройства необходимо вызвать метод:
    - bool AbstractSerial::open(OpenMode mode) - открывает последовательное устройство.\n
    При успешном открытии этот метод вернет \b true.
    \note Устройство всегда открывается в монопольном режиме, т.е. другие процессы не смогут получить доступ к этому устройству.

    Класс AbstractSerial поддерживает только такие опции открытия как:
    OpenMode::ReadOnly, OpenMode::WriteOnly, OpenMode::ReadWrite \n
    Пример:
    \code
        ...
        //this example open device as ReadOnly
        bool ret = serialDevice->open(AbstractSerial::ReadOnly);
        ...
    \endcode

    Для закрытия последовательного устройства необходимо вызвать метод:
    - void AbstractSerial::close() - закрывает последовательное устройство.\n

    После того, как устройство успешно открыто можно приступать к его конфигурированию,
    для чего необходимо воспользоваться следующими методами:
    - bool AbstractSerial::setBaudRate(AbstractSerial::BaudRate) - устанавливает скорость обмена данными.\n
    При успехе метод вернет \b true. \sa setInputBaudRate(BaudRate baudRate), setOutputBaudRate(BaudRate baudRate)
    - bool AbstractSerial::setDataBits(AbstractSerial::DataBits) - устанавливает количество бит данных.\n
    При успехе метод вернет \b true.
    - bool AbstractSerial::setParity(AbstractSerial::Parity) - устанавливает контроль четности.\n
    При успехе метод вернет \b true.
    - bool AbstractSerial::setStopBits(AbstractSerial::StopBits) - устанавливает количество стоп бит.\n
    При успехе метод вернет \b true.
    - bool AbstractSerial::setFlowControl(AbstractSerial::Flow) - устанавливает контроль управления потоком.\n
    При успехе метод вернет \b true.
    - bool AbstractSerial::setCharIntervalTimeout(int msecs) - устанавливает время ожидания прихода символа при чтении данных.\n
    При успехе метод вернет \b true. \n
    \note Т.к. устройство работает в асинхронном режиме,
    то достаточно для уверенного чтения данных подобрать опытным путем только этот параметр таймаута \b msecs.
    Этот параметр является единственным параметром при чтении данных, он только зависит от скорости, и выбирать/рассчитывать его можно опытным путем.
    Обычно для уверенного приема достаточна величина \b 10ms для всех скоростей.
    Это упрощает работу с последовательным устройством.

    .

    Для получения текущей конфигурации экземпляра класса используются методы:
    - QString AbstractSerial::baudRate() const - возвращает понятное для человека значение скорости.\n \sa inputBaudRate() const, outputBaudRate() const
    - QString AbstractSerial::dataBits() const - возвращает понятное для человека кол-во бит данных.\n
    - QString AbstractSerial::parity() const - возвращает понятное для человека значение паритета.\n
    - QString AbstractSerial::stopBits() const - возвращает понятное для человека кол-во стоп-бит.\n
    - QString AbstractSerial::flowControl() const - возвращает понятное для человека значение управления потоком.\n
    - int AbstractSerial::charIntervalTimeout() const - возвращает понятное для человека значение времени ожидания символа.\n
    .

    Для получения списков поддерживаемых экземпляром класса параметров используются методы:
    - QStringList AbstractSerial::listBaudRate() const - возвращает понятный для человека список скоростей.\n
    - QStringList AbstractSerial::listDataBits() const - возвращает понятный для человека список бит данных.\n
    - QStringList AbstractSerial::listParity() const - возвращает понятный для человека список паритетов.\n
    - QStringList AbstractSerial::listStopBits() const - возвращает понятный для человека список стоп-бит.\n
    - QStringList AbstractSerial::listFlowControl() const - возвращает понятный для человека список управления потоком.\n
    .

    Для чтения данных из порта можно использовать любой метод из:
    - qint64 AbstractSerial::read(char *data, qint64 maxSize)
    - QByteArray AbstractSerial::read(qint64 maxSize)
    - QByteArray AbstractSerial::readAll()
    .

    \note Чтение данных осуществляется таким образом, чтобы не терялись данные!

    Для записи данных в порт можно использовать любой метод из:
    - qint64 AbstractSerial::write(const char *data, qint64 maxSize);
    - qint64 AbstractSerial::write(const char *data);
    - qint64 AbstractSerial::write(const QByteArray &byteArray);
    .

    \note Теоретически размер записываемых данных ограничен только лишь максимальным значением maxSize или
    максимальным размером QByteArray. После записи в устройство всех данных излучается сигнал bytesWritten(qint64 bytes).

    Для ожидания прихода данных в последовательное устройство используется метод:
    - bool AbstractSerial::waitForReadyRead(int msecs) - метод ожидает в течении времени msecs прихода в приемный буфер
    последовательного устройства хотя бы одного байта данных.\n
    При успешном завершении ожидания (когда байт пришел до таймаута \b msecs) метод излучает сигнал readyRead() и
    возвращает \b true.

    Для очистки буферов последовательного устройства используются методы:
    - bool AbstractSerial::flush() - принудительно ожидает завершения операций чтения/записи и по их окончании
    очищает очереди ввода/вывода.
    При успехе метод вернет \b true.
    - bool AbstractSerial::reset() - не ожидая завершения операций ввода/вывода очищает буферы последовательного
    устройства.
    При успехе метод вернет \b true.

    Для управления линиями DTR и RTS используются методы:
    - bool AbstractSerial::setDtr(bool set) - устанавливает/сбрасывает линию DTR.\n
    При успехе метод вернет \b true.
    - bool AbstractSerial::setRts(bool set) - устанавливает/сбрасывает линию RTS.\n
    При успехе метод вернет \b true.

    Для получения статусов линий CTS, DSR, DCD, RI, RTS, DTR, ST, SR используется метод:
    - ulong AbstractSerial::lineStatus() - возвращает закодированные значения статусов линий побитовым \b ИЛИ
    (см. enum AbstractSerial::LineStatus). \n

    \verbatim
        CTS  0x01
        DSR  0x02
        DCD  0x04
        RI   0x08
        RTS  0x10
        DTR  0x20
        ST   0x40
        SR   0x80
    \endverbatim

    Для управления разрывами линии Tx используются методы:
    - bool AbstractSerial::sendBreak(int duration) - передает в Tx поток нулевых битов в течении определенного времени
    - bool AbstractSerial::setBreak(bool set) - включает/отключает передачу в Tx нулевых битов
    .

    \note При успехе методы возвращают \b true.

    Для получения текущего количества байт в приемном буфере последовательного устройства готовых для чтения
    используется метод:
    - qint64 AbstractSerial::bytesAvailable() - получает текущее количество байт в приемном буфере.\n

    Для включения/отключения режима излучения сигнала signalStatus() используется метод:
    - void AbstractSerial::enableEmitStatus(bool enable). \n
    .

    \note Для получения списка имен последовательных устройств доступных в системе,
    а также слежения за ними используйте класс: SerialDeviceWatcher. \n


    Класс AbstractSerial реализует следующие сигналы:
    - void AbstractSerial::readyRead() - излучается \b ОДИН раз при приходе в приемный буфер устройства
    хотя бы одного байта данных.\n
    Событие о приходе данных контролируется в цикле сообщений.
    Если приходит следующая часть данных (или один байт) - то сигнал повторно не излучается!\n
    Для того, чтобы сигнал вновь излучился необходимо прочитать из буфера данные (например метод read())
    или сбросить буффер (метод reset())!\n
    Также этот сигнал излучается каждый раз при успешном вызове метода waitForReadyRead().
    - void AbstractSerial::bytesWritten(qint64 bytes) - излучается при успешном завершении метода \b write().\n
    - void AbstractSerial::signalStatus(const QString &status, QDateTime current) - несет информацию о текущем статусе
    последовательного устройства, а также времени и дате возникновения статуса.
    Излучение сигнала можно включать/отключать при вызове: AbstractSerial::enableEmitStatus(bool enable).\n
    Этот сигнал излучается всякий раз при следующих действиях:
        - при выполнении методов open(), close()
        - при ошибках при конфигурировании
        - при ошибках чтении/записи и т.д.

    \n
    \n
    \n

    \warning
    - В этой аннотации приведены только основные методы для работы с классом (т.е. не все), поэтому
    ознакомьтесь с полным перечнем методов самостоятельно! :)
    - По умолчанию класс сконфигурирован таким образом, чтобы записывать в консоль отладочные сообщения.
    Для отключения вывода этих сообщений в консоль - необходимо закомментировать в *.cpp файлах строчки вида:
    \code
        ...
        //#define NATIVESERIALENGINE_DEBUG
        ...
    \endcode
    и заново пересобрать библиотеку.

    \brief \~english Class AbstractSerial provides the minimum and adequate interface for working with serial devices.

    \~english Class to work with serial devices (ports).
    This class is a cross-platform low-level library, which is based on Qt4, and
    respectively, can be used to create applications Qt4.\n

    Implementation of this class structure is broadly similar to the implementation of the supply QAbstractSocket Qt4 and
    corresponds to the ideology of public/private classes (pimpl).\n

    The work of this class of serial device is in asynchronous mode.
    Here, asynchronous mode refers to the fact that the very serial device called
    in non-blocking mode (but in the end class methods anyway awaiting completion of asynchronous I/O!).
    Ie with large volumes of data transmitted through the device when using the GUI may
    GUI freezes during the processing methods of reading/writing data, so it is recommended to create
    object in another thread.\n

    When used in a variety of operating systems - used their native system API:
    - On Windows NT/2K/XP family, etc. use Win32 API.
    - In the POSIX like OS system calls used by POSIX.

    Getting Started with the class should begin with the creation of an object instance AbstractSerial.\n
    Example:
    \code
        ...
        AbstractSerial *serialDevice = new AbstractSerial(this)
        ...
    \endcode

    Also by default, be assigned to the object first name actually exists in the system device.

    \verbatim
        Operating system:          The device name default:
        Windows                    "COM1"
        IRIX                       "/dev/ttyf1"
        HPUX                       "/dev/tty1p0"
        SOLARIS                    "/dev/ttya"
        FREEBSD                    "/dev/ttyd1"
        LINUX                      "/dev/ttyS0"
        <default>                  "/dev/ttyS0"
    \endverbatim

    Next, you need a device (object) to assign a name actually exists in the system:
    - void AbstractSerial::setDeviceName(const QString &deviceName).

    To open a serial device, you must call the method:
    - bool AbstractSerial::open(QIODevice::OpenMode).
    If successful, the opening of this method returns \b true.
    \note The device is always open in exclusive mode, ie other processes can not access the device.

    Class AbstractSerial supports only option open as:
    OpenMode::ReadOnly, OpenMode::WriteOnly, OpenMode::ReadWrite \n
    Example:
    \code
        ...
        //this example open device as ReadOnly
        bool ret = serialDevice->open(AbstractSerial::ReadOnly);
        ...
    \endcode

    To close the serial device need to call the method:
    - void AbstractSerial::close()

    Once the device successfully opened, you are ready to configure it.
    To do this, use the following methods:
    - bool AbstractSerial::setBaudRate(AbstractSerial::BaudRate) - sets the speed of data exchange.\n
    With the success of the method returns \b true. \sa setInputBaudRate(BaudRate baudRate) setOutputBaudRate(BaudRate baudRate)
    - bool AbstractSerial::setDataBits(AbstractSerial::DataBits) - sets the number of data bits.\n
    With the success of the method returns \b true.
    - bool AbstractSerial::setParity(AbstractSerial::Parity) - sets the parity. \n
    With the success of the method returns \b true.
    - bool AbstractSerial::setStopBits(AbstractSerial::StopBits) - sets the number of stop bits. \n
    With the success of the method returns \b true.
    - bool AbstractSerial::setFlowControl(AbstractSerial::Flow) - sets the control flow control.\n
    With the success of the method returns \b true.
    - bool AbstractSerial::setCharIntervalTimeout(int msecs) - sets the waiting time of arrival character when reading data. \n
    With the success of the method returns \b true. \n
    Since device operates in asynchronous mode, it is sufficient for a confident reading of data to select only experienced by this parameter timeout.
    This option is the only option when reading dannh, it just depends on the speed and choose/calculate it can be empirically.
    Typically, for sure receive sufficient magnitude \b 10ms for all speeds.
    It simplifies work with a serial device.

    .

    For the current configuration class instance methods are used:
    - QString AbstractSerial::baudRate() const - returns a human-readable value of speed. \n  \sa inputBaudRate() const, outputBaudRate() const
    - QString AbstractSerial::dataBits() const - returns a human-readable number of data bits. \n
    - QString AbstractSerial::parity() const - returns a human-readable value parity. \n
    - QString AbstractSerial::stopBits() const - returns a human-readable number of stop bits. \n
    - QString AbstractSerial::flowControl() const - returns a human-readable value for flow control. \n
    - int AbstractSerial::charIntervalTimeout() const - returns a human-readable time-out value symbol. \n
    .

    For a list of instance of the parameters used methods:
    - QStringList AbstractSerial::listBaudRate() const - returns human-readable list of speed. \n
    - QStringList AbstractSerial::listDataBits() const - returns human-readable list of data bits. \n
    - QStringList AbstractSerial::listParity() const - returns human-readable list of the parities. \n
    - QStringList AbstractSerial::listStopBits() const - returns human-readable list of stop bits. \n
    - QStringList AbstractSerial::listFlowControl() const - returns human-readable list of flow control. \n
    .

    To read data from the port you can use any method of:
    - qint64 AbstractSerial::read(char *data, qint64 maxSize)
    - QByteArray AbstractSerial::read(qint64 maxSize)
    - QByteArray AbstractSerial::readAll()
    .

    \note reading of data in such a way as to not lose data!

    To write data to the port you can use any method of:
    - qint64 AbstractSerial::write(const char *data, qint64 maxSize);
    - qint64 AbstractSerial::write(const char *data);
    - qint64 AbstractSerial::write(const QByteArray &byteArray);
    .

    \note In theory, the write size is limited to only a maximum value of maxSize or
    maximum size of QByteArray. After write all data in the device emits a signal \b bytesWritten(qint64 bytes).

    In anticipation of the return data in the serial device using the method:
    - bool AbstractSerial::waitForReadyRead(int msecs)\n
    The method waits for the arrival time \b msecs in the receive buffer of the serial device at least one byte of data.
    If successful, the expectation (when B came to a timeout msecs) method emits a signal \b readyRead() and
    returns \b true.

    To clear the buffer of the serial device used methods:
    - bool AbstractSerial::flush() - forced forward to the completion of read/write and at their end
    clears the I/O queues
    - bool AbstractSerial::reset() - not waiting for the completion of I/O clears the serial buffer
    device.

    To control lines DTR and RTS methods are used:
    - bool AbstractSerial::setDtr(bool set)
    - bool AbstractSerial::setRts(bool set)

    If successful, the method returns \b true.

    For the status lines CTS, DSR, DCD, RI, RTS, DTR, ST, SR method is used:
    - ulong AbstractSerial::lineStatus().
    The method returns an encoded value status lines bitwise \b OR (see. enum AbstractSerial::LineStatus). \n

    \verbatim
        CTS     0x01
        DSR     0x02
        DCD     0x04
        RI      0x08
        RTS     0x10
        DTR     0x20
        ST      0x40
        SR      0x80
    \endverbatim

    To control the line breaks Tx methods are used:
    - bool AbstractSerial::sendBreak(int duration) - transfers in Tx stream of zero bits within a certain time
    - bool AbstractSerial::setBreak(bool set) - enables/disables the transfer to the Tx zeros
    .

    \note If the success of the methods return \b true.

    For the current number of bytes in the receive buffer the serial device is ready for reading
    method is used:
    - qint64 AbstractSerial::bytesAvailable().\n

    To enable/disable the radiation signal \b signalStatus() method is used:
    - void AbstractSerial::enableEmitStatus(bool enable).

    For a list of names for serial devices available in the system, as well as tracking them use the class: SerialDeviceWatcher. \n

    Class AbstractSerial implements the following signals:
    - void AbstractSerial::readyRead() - emitted once at the arrival in the receiving buffer device at least one byte of data.\n
    Event of the arrival of data is controlled in the message loop.
    To re-emitted light to read from the buffer data (eg, the method read()) or reset the buffer (the method reset())! \n
    Also, this signal is emitted each time a successful call to the method waitForReadyRead().
    - void AbstractSerial::bytesWritten(qint64 bytes) - emitted when the successful completion write data (eg, the method write()).
    - void AbstractSerial::signalStatus(const QString &status, QDateTime current) - carries information about the current status of the serial
    devices, as well as time and date of status. Is emitted whenever the following actions:
        - The performance of methods open(), close()
        - On error when configuring
        - On error reading/writing, etc.

    The emission signal can be switched on/off when you call: AbstractSerial::enableEmitStatus(bool enable).

    \n
    \n
    \n

    \warning
    - This annotation shows only the basic techniques for working with the class (ie not all), so
    read the full list of methods by yourself! :)
    - By default, the class is configured in such a way as to write to the console debug messages.
    To disable these messages in the console - need to comment out in the *. cpp files, a line like:
    \code
    ...
    //#define NATIVESERIALENGINE_DEBUG
    ...
    \endcode
    and re-rebuild library.


    \version \~ 0.2.0

    \author \~ Denis Shienkov
    - ICQ       : 321789831
    - e-mail    : scapig2@yandex.ru

    \~russian
    \note
    - В версии 0.2.0. работа библиотеки тестировалась автором только с: Windows XP SP3 Pro и ArchLinux x86_64.
    - Библиотекой теоретически должны поддерживаться такие ОС как: все семейство Windows NT/2k/XP , а также POSIX совместимые.
    \~english
    \note
    - In a version 0.2.0. work of library was tested by author only with: Windows XP Sp3 Pro and Archlinux x86_64.
    - In theory must a library be supported such OS as: all of family of Windows Nt/2k/xp, and also POSIX compatible.
*/

#include <QtCore/QStringList>
#include <QtCore/QAbstractEventDispatcher>
#include <QtCore/QObject>

#include "abstractserialengine.h"

//#include <limits.h>

#define ABSTRACTSERIAL_DEBUG

#ifdef ABSTRACTSERIAL_DEBUG
#include <QtCore/QDebug>
#endif

AbstractSerialPrivate::AbstractSerialPrivate()
    :
    emittedReadyRead(false),
    emittedBytesWritten(false),
    emittedStatus(false),
    serialEngine(0)
{
}

AbstractSerialPrivate::~AbstractSerialPrivate()
{
}

void AbstractSerialPrivate::resetSerialLayer()
{
    if (this->serialEngine) {
        this->serialEngine->close();
        delete this->serialEngine;
        this->serialEngine = 0;
    }
}

bool AbstractSerialPrivate::initSerialLayer()
{
    Q_Q(AbstractSerial);
    this->serialEngine = AbstractSerialEngine::createSerialEngine(q);
    if (this->serialEngine) return true;
    return false;
}

void AbstractSerialPrivate::setupSerialNotifiers()
{
    Q_Q(AbstractSerial);
    if (this->serialEngine) {
        switch (q->openMode()) {
            case AbstractSerial::ReadOnly:
            case AbstractSerial::ReadWrite:
                break;
            case AbstractSerial::WriteOnly:
                //TODO: this mode "Write" while not supported notification (i disable it)
                return;
            default: return;
        }
        this->serialEngine->setReadNotificationEnabled(true);
        QObject::connect(this->serialEngine, SIGNAL(readNotification()), q, SLOT(_q_canReadNotification()));
    }
}

void AbstractSerialPrivate::enableSerialNotifiers(bool enable)
{
    if (this->serialEngine) {
        this->serialEngine->setReadNotificationEnabled(enable);
    }
}

void AbstractSerialPrivate::_q_canReadNotification()
{
    Q_Q(AbstractSerial);
    if (!this->emittedReadyRead) {
        this->emittedReadyRead = true;
        emit q->readyRead();
    }
}

void AbstractSerialPrivate::initialiseMap()
{
    //filling m_baudRateMap
#ifndef Q_OS_WIN
    this->m_baudRateMap[AbstractSerial::BaudRate50] = QObject::tr("50 baud");
    this->m_baudRateMap[AbstractSerial::BaudRate75] = QObject::tr("75 baud");
#endif
    this->m_baudRateMap[AbstractSerial::BaudRate110] = QObject::tr("110 baud");
#ifndef Q_OS_WIN
    this->m_baudRateMap[AbstractSerial::BaudRate134] = QObject::tr("134 baud");
    this->m_baudRateMap[AbstractSerial::BaudRate150] = QObject::tr("150 baud");
    this->m_baudRateMap[AbstractSerial::BaudRate200] = QObject::tr("200 baud");
#endif
    this->m_baudRateMap[AbstractSerial::BaudRate300] = QObject::tr("300 baud");
    this->m_baudRateMap[AbstractSerial::BaudRate600] = QObject::tr("600 baud");
    this->m_baudRateMap[AbstractSerial::BaudRate1200] = QObject::tr("1200 baud");
#ifndef Q_OS_WIN
    this->m_baudRateMap[AbstractSerial::BaudRate1800] = QObject::tr("1800 baud");
#endif
    this->m_baudRateMap[AbstractSerial::BaudRate2400] = QObject::tr("2400 baud");
    this->m_baudRateMap[AbstractSerial::BaudRate4800] = QObject::tr("4800 baud");
    this->m_baudRateMap[AbstractSerial::BaudRate9600] = QObject::tr("9600 baud");
#ifdef Q_OS_WIN
    this->m_baudRateMap[AbstractSerial::BaudRate14400] = QObject::tr("14400 baud");
#endif
    this->m_baudRateMap[AbstractSerial::BaudRate19200] = QObject::tr("19200 baud");
    this->m_baudRateMap[AbstractSerial::BaudRate38400] = QObject::tr("38400 baud");
#ifdef Q_OS_WIN
    this->m_baudRateMap[AbstractSerial::BaudRate56000] = QObject::tr("56000 baud");
#endif
    this->m_baudRateMap[AbstractSerial::BaudRate57600] = QObject::tr("57600 baud");
/*
#ifdef Q_OS_WIN
    this->m_baudRateMap[AbstractSerial::BaudRate76800]=QObject::tr("76800 baud");
#endif
*/
    this->m_baudRateMap[AbstractSerial::BaudRate115200] = QObject::tr("115200 baud");
#ifdef Q_OS_WIN
    this->m_baudRateMap[AbstractSerial::BaudRate128000] = QObject::tr("128000 baud");
    this->m_baudRateMap[AbstractSerial::BaudRate256000] = QObject::tr("256000 baud");
#endif

    //filling m_dataBitsMap
    this->m_dataBitsMap[AbstractSerial::DataBits5] = QObject::tr("5 bit");
    this->m_dataBitsMap[AbstractSerial::DataBits6] = QObject::tr("6 bit");
    this->m_dataBitsMap[AbstractSerial::DataBits7] = QObject::tr("7 bit");
    this->m_dataBitsMap[AbstractSerial::DataBits8] = QObject::tr("8 bit");

    //filling m_parityMap
    this->m_parityMap[AbstractSerial::ParityNone] = QObject::tr("None");
    this->m_parityMap[AbstractSerial::ParityOdd] = QObject::tr("Odd");
    this->m_parityMap[AbstractSerial::ParityEven] = QObject::tr("Even");
    this->m_parityMap[AbstractSerial::ParityMark] = QObject::tr("Mark");
    this->m_parityMap[AbstractSerial::ParitySpace] = QObject::tr("Space");

    //filling m_stopBitsMap
    this->m_stopBitsMap[AbstractSerial::StopBits1] = QObject::tr("1");
    this->m_stopBitsMap[AbstractSerial::StopBits1_5] = QObject::tr("1.5");
    this->m_stopBitsMap[AbstractSerial::StopBits2] = QObject::tr("2");

    //filling m_flowMap
    this->m_flowMap[AbstractSerial::FlowControlOff] = QObject::tr("Disable");
    this->m_flowMap[AbstractSerial::FlowControlHardware] = QObject::tr("Hardware");
    this->m_flowMap[AbstractSerial::FlowControlXonXoff] = QObject::tr("Xon/Xoff");
}

/*! Converted from status value to string
*/
QString AbstractSerialPrivate::statusToString(AbstractSerial::Status val) const
{
    switch (val) {
        case AbstractSerial::ENone: return QObject::tr("No errors.");
        case AbstractSerial::ENoneOpen: return QObject::tr("Opened::Device is successfully opened. OK!");
        case AbstractSerial::ENoneClose: return QObject::tr("Closed::Device is successfully closed. OK!");
        case AbstractSerial::ENoneSetBaudRate: return QObject::tr("Controls::Baud rate is successfully set. OK!");
        case AbstractSerial::ENoneSetParity: return QObject::tr("Controls::Parity is successfully set. OK!");
        case AbstractSerial::ENoneSetDataBits: return QObject::tr("Controls::Data bits is successfully set. OK!");
        case AbstractSerial::ENoneSetStopBits: return QObject::tr("Controls::Stop bits is successfully set. OK!");
        case AbstractSerial::ENoneSetFlow: return QObject::tr("Controls::Flow is successfully set. OK!");
        case AbstractSerial::ENoneSetCharTimeout: return QObject::tr("Controls::Char timeout is successfully set. OK!");
        case AbstractSerial::ENoneSetDtr: return QObject::tr("Controls::DTR is successfully changed. OK!");
        case AbstractSerial::ENoneSetRts: return QObject::tr("Controls::RTS is successfully changed. OK!");

        case AbstractSerial::EOpen: return QObject::tr("Error opening. Error!");
        case AbstractSerial::EDeviceIsNotOpen: return QObject::tr("Device is not open. Error!");

        case AbstractSerial::EOpenModeUnsupported: return QObject::tr("Opened::Opened mode unsupported. Error!");
        case AbstractSerial::EOpenModeUndefined: return QObject::tr("Opened::Opened mode undefined. Error!");
        case AbstractSerial::EOpenInvalidFD: return QObject::tr("Opened::Invalid device descriptor. Error!");
        case AbstractSerial::EOpenOldSettingsNotSaved: return QObject::tr("Opened::Fail saved old settings. Error!");
        case AbstractSerial::EOpenGetCurrentSettings: return QObject::tr("Opened::Fail get current settings. Error!");
        case AbstractSerial::EOpenSetDefaultSettings: return QObject::tr("Opened::Fail set default settings. Error!");

        case AbstractSerial::EDeviceIsOpen: return QObject::tr("Device is already open. Error!");

        case AbstractSerial::ECloseSetOldSettings: return QObject::tr("Closed::Fail set old settings. Error!");
        case AbstractSerial::ECloseFD: return QObject::tr("Closed::Fail close device descriptor. Error!");
        case AbstractSerial::EClose: return QObject::tr("Closed::Fail close device. Error!");

        case AbstractSerial::ESetBaudRate: return QObject::tr("Parameters::Set baud rate fail. Error!");
        case AbstractSerial::ESetDataBits: return QObject::tr("Parameters::Set data bits fail. Error!");
        case AbstractSerial::ESetParity: return QObject::tr("Parameters::Set parity fail. Error!");
        case AbstractSerial::ESetStopBits: return QObject::tr("Parameters::Set stop bits fail. Error!");
        case AbstractSerial::ESetFlowControl: return QObject::tr("Parameters::Set flow control fail. Error!");
        case AbstractSerial::ESetCharIntervalTimeout: return QObject::tr("Parameters::Set char interval timeout. Error!");

        case AbstractSerial::EBytesAvailable: return QObject::tr("Controls::Get bytes available fail. Error!");
        case AbstractSerial::ESetDtr: return QObject::tr("Controls::Set DTR fail. Error!");
        case AbstractSerial::ESetRts: return QObject::tr("Controls::Set RTS fail. Error!");
        case AbstractSerial::ELineStatus: return QObject::tr("Controls::Get lines status fail. Error!");
        case AbstractSerial::EWaitReadyReadIO: return QObject::tr("Controls::Wait for ready read from device - i/o problem. Error!");
        case AbstractSerial::EWaitReadyReadTimeout: return QObject::tr("Controls::Wait for ready read timeout. Error!");
        case AbstractSerial::EWaitReadyWriteIO: return QObject::tr("Controls::Wait for bytes writtten to device - i/o problem. Error!");
        case AbstractSerial::EWaitReadyWriteTimeout: return QObject::tr("Controls::Wait for bytes writtten timeout. Error!");
        case AbstractSerial::EReadDataIO: return QObject::tr("Controls::Read data from device - i/o problem. Error!");
        case AbstractSerial::EWriteDataIO: return QObject::tr("Controls::Write data to device - i/o problem. Error!");
        case AbstractSerial::EFlush: return QObject::tr("Controls::Flush fail. Error!");
        case AbstractSerial::ESendBreak: return QObject::tr("Controls::Send break fail. Error!");
        case AbstractSerial::ESetBreak: return QObject::tr("Controls::Set break fail. Error!");

        default: return QObject::tr("AbstractSerial::statusToString(Status val) -> Status mode: %1 undefined. Error!").arg(val);
    }
}


//----------------------------------------------------------------------------------------------------------------------------------------

/*! \fn AbstractSerial::AbstractSerial(QObject *parent)
    \~russian Конструктор по умолчанию. Создает объект, ассоциированный с реальным последовательным устройством в системе.
    Например:
    - для Windows - это COM1, COM2 ... COMn
    - для Linux - это /dev/ttyS0, /dev/ttyS1 и т.д.
    \note  Если не указать (пропустить) в конструкторе параметр deviceName - то по умолчанию устройству присвоится имя = DEFAULT_DEVICE_NAME
    \~english Default constructor. Creates an object, associated with the real serial device in the system.
    For example: for Windows - it COM1, COM2 ... COMn, for Linux - it /dev/ttyS0, /dev/ttyS1 etc
    \note If not to specify (to skip) the parameter of devicename in a constructor - that by default the name will be appropriated a device = DEFAULT_DEVICE_NAME
    \~
    \verbatim
        Operating system:       Name DEFAULT_DEVICE_NAME:
        Windows                 "COM1"
        IRIX                    "/dev/ttyf1"
        HPUX                    "/dev/tty1p0"
        SOLARIS                 "/dev/ttya"
        FREEBSD                 "/dev/ttyd1"
        LINUX                   "/dev/ttyS0"
        <default>               "/dev/ttyS0"
    \endverbatim
*/
AbstractSerial::AbstractSerial(QObject *parent)
    : QObject(parent), d_ptr(new AbstractSerialPrivate())
{
    Q_D(AbstractSerial);
    d->q_ptr = this;
    if (!d->initSerialLayer()) return;
    d->initialiseMap();
}

/*! \fn AbstractSerial::~AbstractSerial()
    \~russian Деструктор по умолчанию.
    \~english Default destructor.
*/
AbstractSerial::~AbstractSerial()
{
    Q_D(AbstractSerial);
    d->resetSerialLayer();
    delete d_ptr;
}

/*! \fn void AbstractSerial::setDeviceName(const QString &deviceName)
    \~russian Присваивает созданному объекту имя последовательного устройства, с которым необходимо работать.
    \param[in] deviceName - имя последовательного устройства, реально имеющегося в системе.\n
    Например: для Windows - это COM1, COM2 ... COMn, для Linux - это /dev/ttyS0, /dev/ttyS1 и т.д.
    \note Использовать этот метод необходимо только ПЕРЕД открытием устройства!
    \~english Appropriates the created object the serial device with which it is necessary to work name.
    \param[in] deviceName - serial device, really present in the system name.\n
    For example: for Windows - it COM1, COM2 ... COMn, for Linux - it /dev/ttyS0, /dev/ttyS1 etc
    \note  Uses this method is necessary only BEFORE opening of device!
*/
void AbstractSerial::setDeviceName(const QString &deviceName)
{
    Q_D(AbstractSerial);
    if (this->isOpen()) {
#if defined (ABSTRACTSERIAL_DEBUG)
    qDebug() << "AbstractSerial::setDeviceName(const QString &deviceName) \n"
                " -> forbidden to set the name for the open device. Warning!";
#endif
        return;
    }
    if (this->isValid())
        d->serialEngine->setDeviceName(deviceName);
}

/*! \fn QString AbstractSerial::deviceName() const
    \~russian Возвращает имя последовательного устройства, с которым сконфигурирован объект.
    \return
    - Имя последовательного устройства, с которым сконфигурирован объект в виде QString
    - 0 в случае ошибки
    \note Использовать этот метод можно в любой момент после создания объекта, т.е. неважно открыто реально устройство или нет.
    \~english Returns the serial device which an object is configured with name.
    \return
    - Serial device with which an object is configured as QString name
    - 0 in the case of error
    \note Uses this method is possible at any moment after creation of object, I.e. it is unimportant openly really device or not.
*/
QString AbstractSerial::deviceName() const
{
    Q_D(const AbstractSerial);
    return (this->isValid()) ? d->serialEngine->deviceName() : QString();
}

/*! \fn void AbstractSerial::setOpenMode(AbstractSerial::OpenMode mode)
    \~russian Устанавливает режим открытия последовательного устройства.
    \param[in] mode - режим открытия (см. AbstractSerial::OpenMode)
    \note Этот метод отработает только если устройство еще не открыто!
    \~english Sets the mode of opening the serial device.
    \param[in] mode - open mode (see. AbstractSerial::OpenMode)
    \note This method can run only if the device is not open!
*/
void AbstractSerial::setOpenMode(AbstractSerial::OpenMode mode)
{
    Q_D(const AbstractSerial);
    if ( this->isValid() && (!this->isOpen()) )
        d->serialEngine->setOpenMode(mode);
}

/*! \fn AbstractSerial::OpenMode AbstractSerial::openMode() const
    \~russian Возвращает текущий режим открытия последовательного устройства
    \return режим открытия
    \~english Returns the current mode of opening the serial device
    \return open mode
*/
AbstractSerial::OpenMode AbstractSerial::openMode() const
{
    Q_D(const AbstractSerial);
    if (this->isValid())
        return d->serialEngine->openMode();
    return AbstractSerial::NotOpen;
}

/*! \fn bool AbstractSerial::open(OpenMode mode)
    \~russian Открывает последовательное устройство.
    \param[in] mode - режим открытия последовательного устройства
    \return
    - true - при успехе
    - false - в случае ошибки
    \note
    - После того, как устройство успешно открыто - можно приступать к его конфигурированию
    - Метод не отработает если устройство уже было открыто
    \~english Opens a serial device.
    \param[in] mode - mode of opening of serial device
    \return
    - true - at success
    - false - in the case of error
    \note
    - Since a device is successful openly - it is possible to proceed to his configuring
    - The method does not run for when the device has been opened
*/
bool AbstractSerial::open(OpenMode mode)
{
    Q_D(AbstractSerial);
    if (!this->isOpen()) {
        if (this->isValid()) {
            if (d->serialEngine->open(mode)) {
                if (QAbstractEventDispatcher::instance(thread()))
                    d->setupSerialNotifiers();
                this->emitStatusString(ENoneOpen);
                return true;
            }
        }//isValid
        this->emitStatusString(EOpen);
        return false;
    }
    this->emitStatusString(EDeviceIsOpen);
    return false;
}

/*! \fn bool AbstractSerial::isOpen() const
    \~russian Возвращает статус последовательного устройства: открыто/закрыто
    \return
    - true - устройство открыто
    - false - устройство закрыто
    \~english Returns the status of the serial device: open / closed
    \return
    - true - the device is open
    - false - the device is closed
*/
bool AbstractSerial::isOpen() const
{
    Q_D(const AbstractSerial);
    if (this->isValid()) {
        return d->serialEngine->isOpen();
    }
    return false;
}

/*! \fn void AbstractSerial::close()
    \~russian Закрывает последовательное устройство.
    \~english Closes a serial device
*/
void AbstractSerial::close()
{
    Q_D(AbstractSerial);
    if (this->isOpen()) {
        d->serialEngine->close();
        this->emitStatusString(ENoneClose);
    }
    else
        this->emitStatusString(EClose);
}

/*! \fn bool AbstractSerial::setBaudRate(BaudRate baudRate)
    \~russian Устанавливает для последовательного устройства скорость в бодах.
    \note
    - в Windows - этот метод устанавливает только один тип скорости
    - в *.nix - этот метод устанавливает входящую скорость и исходящую скорость одинаковыми.
    \param[in] baudRate - желаемая скорость последовательного устройства в бодах из перечисления.
    \return
    - true - при успехе
    - false - в случае ошибки
    \note Использовать этот метод можно \b ТОЛЬКО \b ПОСЛЕ \b ТОГО \b КАК \b УСТРОЙСТВО \b ОТКРЫЛИ! (иначе параметр \b baudRate будет игнорироваться)
    \~english Sets a baud rate for a serial device
    \note
    - in Windows - this method sets only one type of speed.
    - in *. nix - this method sets the speed of input and output speed of the equal.
    \param[in] baudRate -  desired baud rate of serial device from enumeration.
    \return
    - true - at success
    - false - at an error
    \note Uses this method is possible ONLY Since DEVICE was OPENED! (the parameter of BaudRate will be ignored otherwise)
*/
bool AbstractSerial::setBaudRate(BaudRate baudRate)
{
    Q_D(AbstractSerial);
    if (this->isOpen()) {
        bool ret = d->serialEngine->setBaudRate(baudRate);
        (ret) ? this->emitStatusString(ENoneSetBaudRate) : this->emitStatusString(ESetBaudRate);
        return ret;
    }
    this->emitStatusString(EDeviceIsNotOpen);
    return false;
}

/*! \fn bool AbstractSerial::setInputBaudRate(BaudRate baudRate)
    \~russian Устанавливает для последовательного устройства только входящую скорость в бодах.
    \param[in] baudRate - желаемая входящая скорость последовательного устройства в бодах из перечисления.
    \return
    - true - при успехе
    - false - в случае ошибки
    .
    \note
    - в Windows - этот метод всегда возвращает false!
    - использовать этот метод можно \b ТОЛЬКО \b ПОСЛЕ \b ТОГО \b КАК \b УСТРОЙСТВО \b ОТКРЫЛИ! (иначе параметр \b baudRate будет игнорироваться)
    \~english Sets a only input baud rate for a serial device
    \param[in] baudRate -  desired input baud rate of serial device from enumeration.
    \return
    - true - at success
    - false - at an error
    .
    \note
    - in Windows - this method always returns false!
    - uses this method is possible ONLY Since DEVICE was OPENED! (the parameter of BaudRate will be ignored otherwise)
*/
bool AbstractSerial::setInputBaudRate(BaudRate baudRate)
{
    Q_D(AbstractSerial);
    if (this->isOpen()) {
        bool ret = d->serialEngine->setInputBaudRate(baudRate);
        (ret) ? this->emitStatusString(ENoneSetBaudRate) : this->emitStatusString(ESetBaudRate);
        return ret;
    }
    this->emitStatusString(EDeviceIsNotOpen);
    return false;
}

/*! \fn bool AbstractSerial::setOutputBaudRate(BaudRate baudRate)
    \~russian Устанавливает для последовательного устройства только исходящую скорость в бодах.
    \param[in] baudRate - желаемая исходящая скорость последовательного устройства в бодах из перечисления.
    \return
    - true - при успехе
    - false - в случае ошибки
    .
    \note
    - в Windows - этот метод всегда возвращает false!
    - использовать этот метод можно \b ТОЛЬКО \b ПОСЛЕ \b ТОГО \b КАК \b УСТРОЙСТВО \b ОТКРЫЛИ! (иначе параметр \b baudRate будет игнорироваться)
    \~english Sets a only output baud rate for a serial device
    \param[in] baudRate -  desired output baud rate of serial device from enumeration.
    \return
    - true - at success
    - false - at an error
    .
    \note
    - in Windows - this method always returns false!
    - uses this method is possible ONLY Since DEVICE was OPENED! (the parameter of BaudRate will be ignored otherwise)
*/
bool AbstractSerial::setOutputBaudRate(BaudRate baudRate)
{
    Q_D(AbstractSerial);
    if (this->isOpen()) {
        bool ret = d->serialEngine->setOutputBaudRate(baudRate);
        (ret) ? this->emitStatusString(ENoneSetBaudRate) : this->emitStatusString(ESetBaudRate);
        return ret;
    }
    emitStatusString(EDeviceIsNotOpen);
    return false;
}

/*! \fn bool AbstractSerial::setBaudRate(const QString &baudRate)
    \overload
    \~russian Устанавливает для последовательного устройства скорость в бодах.
    \param[in] baudRate - желаемая скорость последовательного устройства в бодах в виде строки.
    \return
    - true - при успехе
    - false - в случае ошибки
    \note Использовать этот метод можно \b ТОЛЬКО \b ПОСЛЕ \b ТОГО \b КАК \b УСТРОЙСТВО \b ОТКРЫЛИ! (иначе параметр \b baudRate будет игнорироваться)
    \~english Sets a baud rate for a serial device
    \param[in] baudRate -  desired baud rate of serial device as an string.
    \return
    - true - at success
    - false - at an error
    \note Uses this method is possible ONLY Since DEVICE was OPENED! (the parameter of BaudRate will be ignored otherwise)
*/
bool AbstractSerial::setBaudRate(const QString &baudRate)
{
    Q_D(AbstractSerial);
    int res = d->m_baudRateMap.key(baudRate);
    if (!res) {
#if defined (ABSTRACTSERIAL_DEBUG)
    qDebug() << "AbstractSerial::setBaudRate(const QString &baudRate) \n"
                " -> returned: false because input parameter speed:" <<  baudRate << "\n is not defined in QMap. Error!";
#endif
        this->emitStatusString(ENoneSetBaudRate);
        return false;
    }
    return this->setBaudRate(BaudRate(res));
}

/*! \fn bool AbstractSerial::setInputBaudRate(const QString &baudRate)
    \overload
    \~russian Устанавливает для последовательного устройства только входящую скорость в бодах.
    \param[in] baudRate - желаемая входящая скорость последовательного устройства в бодах бодах в виде строки.
    \return
    - true - при успехе
    - false - в случае ошибки
    .
    \note
    - в Windows - этот метод всегда возвращает false!
    - использовать этот метод можно \b ТОЛЬКО \b ПОСЛЕ \b ТОГО \b КАК \b УСТРОЙСТВО \b ОТКРЫЛИ! (иначе параметр \b baudRate будет игнорироваться)
    \~english Sets a only input baud rate for a serial device
    \param[in] baudRate -  desired input baud rate of serial device as an string.
    \return
    - true - at success
    - false - at an error
    .
    \note
    - in Windows - this method always returns false!
    - uses this method is possible ONLY Since DEVICE was OPENED! (the parameter of BaudRate will be ignored otherwise)
*/
bool AbstractSerial::setInputBaudRate(const QString &baudRate)
{
    Q_D(AbstractSerial);
    int res = d->m_baudRateMap.key(baudRate);
    if (!res) {
#if defined (ABSTRACTSERIAL_DEBUG)
    qDebug() << "AbstractSerial::setInputBaudRate(const QString &baudRate) \n"
                " -> returned: false because input parameter speed:" <<  baudRate << "\n is not defined in QMap. Error!";
#endif
        this->emitStatusString(ENoneSetBaudRate);
        return false;
    }
    return this->setInputBaudRate(BaudRate(res));
}

/*! \fn bool AbstractSerial::setOutputBaudRate(const QString &baudRate)
    \overload
    \~russian Устанавливает для последовательного устройства только исходящую скорость в бодах.
    \param[in] baudRate - желаемая исходящая скорость последовательного устройства в бодах бодах в виде строки.
    \return
    - true - при успехе
    - false - в случае ошибки
    .
    \note
    - в Windows - этот метод всегда возвращает false!
    - использовать этот метод можно \b ТОЛЬКО \b ПОСЛЕ \b ТОГО \b КАК \b УСТРОЙСТВО \b ОТКРЫЛИ! (иначе параметр \b baudRate будет игнорироваться)
    \~english Sets a only input baud rate for a serial device
    \param[in] baudRate -  desired input baud rate of serial device as an string.
    \return
    - true - at success
    - false - at an error
    .
    \note
    - in Windows - this method always returns false!
    - uses this method is possible ONLY Since DEVICE was OPENED! (the parameter of BaudRate will be ignored otherwise)
*/
bool AbstractSerial::setOutputBaudRate(const QString &baudRate)
{
    Q_D(AbstractSerial);
    int res = d->m_baudRateMap.key(baudRate);
    if (!res) {
#if defined (ABSTRACTSERIAL_DEBUG)
    qDebug() << "AbstractSerial::setOutputBaudRate(const QString &baudRate) \n"
                " -> returned: false because input parameter speed:" <<  baudRate << "\n is not defined in QMap. Error!";
#endif
        this->emitStatusString(ENoneSetBaudRate);
        return false;
    }
    return this->setOutputBaudRate(BaudRate(res));
}

/*! \fn QString AbstractSerial::baudRate() const
    \~russian Возвращает скорость в бодах в виде строки с которой сконфигурировано последовательное устройство.
    \return
    - Скорость в виде строки QString - если входящая скорость и исходящая скорость одинаковые.
    - Скорость в виде строки QString "Baud rate undefined" - если входящая скорость и исходящая скорость разные.
    - 0 - в случае ошибки
    .
    \note В OS Windows всегда (?) входящая и исходящая скорость одинаковые.
    \~english  Returns baud rate as a string which a serial device is configured with.
    \return
    - Speed in a string QString - if the speed of input and speed of output is equal.
    - Speed in a string QString "Baud rate undefined" - if the speed of input and speed of output is not equal.
    - 0 - in case of error
    .
    \note In OS Windows always (?) input and output speed is the equal.
*/
QString AbstractSerial::baudRate() const
{
    Q_D(const AbstractSerial);
    if (this->isValid()) {
        AbstractSerial::BaudRate r = d->serialEngine->baudRate();
        if (r == AbstractSerial::BaudRateUndefined)
            return QObject::tr("Baud rate undefined");

        QString res = d->m_baudRateMap.value(r);

        if (res.isEmpty()) {
#if defined (ABSTRACTSERIAL_DEBUG)
    qDebug() << "AbstractSerial::baudRate() -> hash is empty. Error!";
#endif
        }
        return res;
    }
#if defined (ABSTRACTSERIAL_DEBUG)
    qDebug() << "AbstractSerial::baudRate() -> object AbstractSerial is invalid. Error!";
#endif
    return QString();
}

/*! \fn QString AbstractSerial::inputBaudRate() const
    \~russian Возвращает входящую скорость в бодах в виде строки с которой сконфигурировано последовательное устройство.
    \return
    - Скорость в виде строки QString - если входящая скорость определена
    - Скорость в виде строки QString "Baud rate undefined" - если входящая скорость неопределена
    - 0 - в случае ошибки
    .
    \note В OS Windows всегда (?) входящая и исходящая скорость одинаковые.
    \~english Returns input baud rate as a string which a serial device is configured with.
    \return
    - Speed in a string QString - if the input rate is determined
    - Speed in a string QString "Baud rate undefined" - if the input rate is undefined
    - 0 - in case of error
    .
    \note In OS Windows always (?) input and output speed is the equal.
*/
QString AbstractSerial::inputBaudRate() const
{
    Q_D(const AbstractSerial);
    if (this->isValid()) {
        AbstractSerial::BaudRate r = d->serialEngine->inputBaudRate();
        if (r == AbstractSerial::BaudRateUndefined)
            return QObject::tr("Baud rate undefined");

        QString res = d->m_baudRateMap.value(r);

        if (res.isEmpty()) {
#if defined (ABSTRACTSERIAL_DEBUG)
    qDebug() << "AbstractSerial::inputBaudRate() -> hash is empty. Error!";
#endif
        }
        return res;
    }
#if defined (ABSTRACTSERIAL_DEBUG)
    qDebug() << "AbstractSerial::inputBaudRate() -> object AbstractSerial is invalid. Error!";
#endif
    return QString();
}

/*! \fn QString AbstractSerial::outputBaudRate() const
    \~russian Возвращает исходящую скорость в бодах в виде строки с которой сконфигурировано последовательное устройство.
    \return
    - Скорость в виде строки QString - если исходящая скорость определена
    - Скорость в виде строки QString "Baud rate undefined" - если исходящая скорость неопределена
    - 0 - в случае ошибки
    .
    \note В OS Windows всегда (?) входящая и исходящая скорость одинаковые.
    \~english Returns input baud rate as a string which a serial device is configured with.
    \return
    - Speed in a string QString - if the output rate is determined
    - Speed in a string QString "Baud rate undefined" - if the output rate is undefined
    - 0 - in case of error
    .
    \note In OS Windows always (?) input and output speed is the equal.
*/
QString AbstractSerial::outputBaudRate() const
{
    Q_D(const AbstractSerial);
    if (this->isValid()) {
        AbstractSerial::BaudRate r = d->serialEngine->outputBaudRate();
        if (r == AbstractSerial::BaudRateUndefined)
            return QObject::tr("Baud rate undefined");

        QString res = d->m_baudRateMap.value(r);

        if (res.isEmpty()) {
#if defined (ABSTRACTSERIAL_DEBUG)
    qDebug() << "AbstractSerial::outputBaudRate() -> hash is empty. Error!";
#endif
        }
        return res;
    }
#if defined (ABSTRACTSERIAL_DEBUG)
    qDebug() << "AbstractSerial::outputBaudRate() -> object AbstractSerial is invalid. Error!";
#endif
    return QString();
}

/*! \fn QStringList AbstractSerial::listBaudRate() const
    \~russian Возвращает текстовый список всех скоростей, поддерживаемых классом AbstractSerial.
    \return Список скоростей в виде QStringList
    \~english Returns the text list of all of speeds (baud rates), supported the class of Abstractserial.
    \return List of speeds (baud rates) as QStringlist
*/
QStringList AbstractSerial::listBaudRate() const
{
    Q_D(const AbstractSerial);
    static const QStringList list = d->m_baudRateMap.values();
    return list;
}

/*! \fn QMap<AbstractSerial::BaudRate, QString> AbstractSerial::baudRateMap() const
    \~russian Возвращает мапу всех скоростей поддерживаемых классом Abstractserial. Используется для заполнения QComboBox.
    \return Список скоростей как QMap<AbstractSerial::BaudRate, QString>
    \~english Returns the map of all speeds (baud rates), supported by the class Abstractserial. Useful for filling up QComboBox.
    \return List of speeds (baud rates) as QMap<AbstractSerial::BaudRate, QString>
*/
QMap<AbstractSerial::BaudRate, QString> AbstractSerial::baudRateMap() const
{
    Q_D(const AbstractSerial);
    return d->m_baudRateMap;
}

/*! \fn bool AbstractSerial::setDataBits(DataBits dataBits)
    \~russian Устанавливает для последовательного устройства количество бит данных.
    \param[in] dataBits - желаемое количество бит данных последовательного устройства из перечисления.
    \return
    - true - при успехе
    - false - в случае ошибки
    \note Использовать этот метод можно \b ТОЛЬКО \b ПОСЛЕ \b ТОГО \b КАК \b УСТРОЙСТВО \b ОТКРЫЛИ! (иначе параметр \b dataBits будет игнорироваться)
    \~english Sets a data bits for a serial device
    \param[in] dataBits - desired data bits of serial device from enumeration.
    \return
    - true - at success
    - false - at an error
    \note Uses this method is possible ONLY Since DEVICE was OPENED! (the parameter of dataBits will be ignored otherwise)
*/
bool AbstractSerial::setDataBits(DataBits dataBits)
{
    Q_D(AbstractSerial);
    if (this->isOpen()) {
        bool ret = d->serialEngine->setDataBits(dataBits);
        (ret) ? this->emitStatusString(ENoneSetDataBits) : this->emitStatusString(ESetDataBits);
        return ret;
    }
    this->emitStatusString(EDeviceIsNotOpen);
    return false;
}

/*! \fn bool AbstractSerial::setDataBits(const QString &dataBits)
    \overload
    \~russian Устанавливает для последовательного устройства количество бит данных.
    \param[in] dataBits - желаемое количество бит данных последовательного устройства в виде строки
    \return
    - true - при успехе
    - false - в случае ошибки
    \note Использовать этот метод можно \b ТОЛЬКО \b ПОСЛЕ \b ТОГО \b КАК \b УСТРОЙСТВО \b ОТКРЫЛИ! (иначе параметр \b dataBits будет игнорироваться)
    \~english Sets a data bits for a serial device
    \param[in] dataBits -  desired data bits of serial device as an string
    \return
    - true - at success
    - false - at an error
    \note Uses this method is possible ONLY Since DEVICE was OPENED! (the parameter of dataBits will be ignored otherwise)
*/
bool AbstractSerial::setDataBits(const QString &dataBits)
{
    Q_D(const AbstractSerial);
    int res = d->m_dataBitsMap.key(dataBits);
    if (!res) {
#if defined (ABSTRACTSERIAL_DEBUG)
    qDebug() << "AbstractSerial::setDataBits(const QString &dataBits) \n"
                " -> returned: false because input parameter databits:" <<  dataBits << "\n is not defined in QMap. Error!";
#endif
        this->emitStatusString(ENoneSetDataBits);
        return false;
    }
    return this->setDataBits(DataBits(res));
}

/*! \fn QString AbstractSerial::dataBits() const
    \~russian Возвращает количество бит данных в виде строки с которым сконфигурировано последовательное устройство.
    \return
    - Количество бит данных в виде строки QString
    - 0 - в случае ошибки
    \~english  Returns data bits as a string which a serial device is configured with.
    \return
    - data bits as an string QString
    - 0 - in the case of error
*/
QString AbstractSerial::dataBits() const
{
    Q_D(const AbstractSerial);
    if (this->isValid()) {
        AbstractSerial::DataBits r = d->serialEngine->dataBits();
        if (r == AbstractSerial::DataBitsUndefined)
            return QObject::tr("Data bits undefined");

        QString res = d->m_dataBitsMap.value(r);

        if (res.isEmpty()) {
#if defined (ABSTRACTSERIAL_DEBUG)
    qDebug() << "AbstractSerial::dataBits() -> hash is empty. Error!";
#endif
        }
        return res;
    }
#if defined (ABSTRACTSERIAL_DEBUG)
    qDebug() << "AbstractSerial::dataBits() -> object AbstractSerial is invalid. Error!";
#endif
    return QString();
}

/*! \fn QStringList AbstractSerial::listDataBits() const
    \~russian Возвращает текстовый список всех типов бит данных, поддерживаемых классом AbstractSerial.
    \return Список бит данных в виде QStringList
    \~english Returns the text list of all of data bits, supported the class of Abstractserial.
    \return List of data bits as QStringlist
*/
QStringList AbstractSerial::listDataBits() const
{
    Q_D(const AbstractSerial);
    static const QStringList list = d->m_dataBitsMap.values();
    return list;
}

/*! \fn QMap<AbstractSerial::DataBits, QString> AbstractSerial::dataBitsMap() const
    \~russian Возвращает мапу всех видов бит данных, поддерживаемых классом AbstractSerial. Используется для заполнения QComboBox.
    \return Список типов бит данных как QMap<AbstractSerial::DataBits, QString>
    \~english Returns the map of all data bit options, supported by the class Abstractserial. Useful for filling up QComboBox.
    \return List of data bit options as QMap<AbstractSerial::DataBits, QString>
*/
QMap<AbstractSerial::DataBits, QString> AbstractSerial::dataBitsMap() const
{
    Q_D(const AbstractSerial);
    return d->m_dataBitsMap;
}

/*! \fn bool AbstractSerial::setParity(Parity parity)
    \~russian Устанавливает для последовательного устройства тип контроля четности.
    \param[in] parity - желаемый тип контроля четности последовательного устройства из перечисления.
    \return
    - true - при успехе
    - false - в случае ошибки
    \note Использовать этот метод можно \b ТОЛЬКО \b ПОСЛЕ \b ТОГО \b КАК \b УСТРОЙСТВО \b ОТКРЫЛИ! (иначе параметр \b parity будет игнорироваться)
    \~english Sets a parity for a serial device
    \param[in] parity - desired parity of serial device from enumeration.
    \return
    - true - at success
    - false - at an error
    \note Uses this method is possible ONLY Since DEVICE was OPENED! (the parameter of parity will be ignored otherwise)
*/
bool AbstractSerial::setParity(Parity parity)
{
    Q_D(AbstractSerial);
    if (this->isOpen()) {
        bool ret = d->serialEngine->setParity(parity);
        (ret) ? this->emitStatusString(ENoneSetParity) : this->emitStatusString(ESetParity);
        return ret;
    }
    this->emitStatusString(EDeviceIsNotOpen);
    return false;
}

/*! \fn bool AbstractSerial::setParity(const QString &parity)
    \overload
    \~russian Устанавливает для последовательного устройства тип контроля четности.
    \param[in] parity - желаемый тип контроля четности последовательного устройства в виде строки
    \return
    - true - при успехе
    - false - в случае ошибки
    \note Использовать этот метод можно \b ТОЛЬКО \b ПОСЛЕ \b ТОГО \b КАК \b УСТРОЙСТВО \b ОТКРЫЛИ! (иначе параметр \b parity будет игнорироваться)
    \~english Sets a parity for a serial device
    \param[in] parity -  desired parity of serial device as an string
    \return
    - true - at success
    - false - at an error
    \note Uses this method is possible ONLY Since DEVICE was OPENED! (the parameter of parity will be ignored otherwise)
*/
bool AbstractSerial::setParity(const QString &parity)
{
    Q_D(const AbstractSerial);
    int res = d->m_parityMap.key(parity);
    if (!res) {
#if defined (ABSTRACTSERIAL_DEBUG)
    qDebug() << "AbstractSerial::setParity(const QString &parity) \n"
                " -> returned: false because input parameter parity:" <<  parity << "\n is not defined in QMap. Error!";
#endif
        this->emitStatusString(ENoneSetParity);
        return false;
    }
    return this->setParity(Parity(res));
}

/*! \fn QString AbstractSerial::parity() const
    \~russian Возвращает тип контроля четности в виде строки с которым сконфигурировано последовательное устройство.
    \return
    - Тип контроля четности в виде строки QString
    - 0 - в случае ошибки
    \~english  Returns parity as a string which a serial device is configured with.
    \return
    - parity as an string QString
    - 0 - in the case of error
*/
QString AbstractSerial::parity() const
{
    Q_D(const AbstractSerial);
    if (this->isValid()) {
        AbstractSerial::Parity r = d->serialEngine->parity();
        if (r == AbstractSerial::ParityUndefined)
            return QObject::tr("Parity undefined");

        QString res = d->m_parityMap.value(r);

        if (res.isEmpty()) {
#if defined (ABSTRACTSERIAL_DEBUG)
    qDebug() << "AbstractSerial::parity() -> hash is empty. Error!";
#endif
        }
        return res;
    }
#if defined (ABSTRACTSERIAL_DEBUG)
    qDebug() << "AbstractSerial::parity() -> object AbstractSerial is invalid. Error!";
#endif
    return QString();
}

/*! \fn QStringList AbstractSerial::listParity() const
    \~russian Возвращает текстовый список всех типов контроля четности , поддерживаемых классом AbstractSerial.
    \return Список контроля четности в виде QStringList
    \~english Returns the text list of all of paritys, supported the class of Abstractserial.
    \return List of paritys as QStringlist
*/
QStringList AbstractSerial::listParity() const
{
    Q_D(const AbstractSerial);
    static const QStringList list = d->m_parityMap.values();
    return list;
}

/*! \fn QMap<AbstractSerial::Parity, QString> AbstractSerial::parityMap() const
    \~russian Возвращает мапу всех видов паритета, поддерживаемых классом AbstractSerial. Используется для заполнения QComboBox.
    \return Список паритетов как QMap<AbstractSerial::Parity, QString>
    \~english Returns the map of all parity options, supported by the class Abstractserial. Useful for filling up QComboBox.
    \return Map of parity options as QMap<AbstractSerial::Parity, QString>
*/
QMap<AbstractSerial::Parity, QString> AbstractSerial::parityMap() const
{
    Q_D(const AbstractSerial);
    return d->m_parityMap;
}

/*! \fn bool AbstractSerial::setStopBits(StopBits stopBits)
    \~russian Устанавливает для последовательного устройства количество стоп-бит.
    \param[in] stopBits - желаемое количество стоп-бит последовательного устройства из перечисления.
    \return
    - true - при успехе
    - false - в случае ошибки
    \note Использовать этот метод можно \b ТОЛЬКО \b ПОСЛЕ \b ТОГО \b КАК \b УСТРОЙСТВО \b ОТКРЫЛИ! (иначе параметр \b stopBits будет игнорироваться)
    \~english Sets a stop bits for a serial device
    \param[in] stopBits - desired stop bits of serial device from enumeration.
    \return
    - true - at success
    - false - at an error
    \note Uses this method is possible ONLY Since DEVICE was OPENED! (the parameter of stopBits will be ignored otherwise)
*/
bool AbstractSerial::setStopBits(StopBits stopBits)
{
    Q_D(AbstractSerial);
    if (this->isOpen()) {
        bool ret = d->serialEngine->setStopBits(stopBits);
        (ret) ? this->emitStatusString(ENoneSetStopBits) : this->emitStatusString(ESetStopBits);
        return ret;
    }
    this->emitStatusString(EDeviceIsNotOpen);
    return false;
}

/*! \fn bool AbstractSerial::setStopBits(const QString &stopBits)
    \overload
    \~russian Устанавливает для последовательного устройства количество стоп-бит.
    \param[in] stopBits - желаемое количество стоп-бит последовательного устройства в виде строки
    \return
    - true - при успехе
    - false - в случае ошибки
    \note Использовать этот метод можно \b ТОЛЬКО \b ПОСЛЕ \b ТОГО \b КАК \b УСТРОЙСТВО \b ОТКРЫЛИ! (иначе параметр \b stopBits будет игнорироваться)
    \~english Sets a stop bits for a serial device
    \param[in] stopBits -  desired stop bits of serial device as an string
    \return
    - true - at success
    - false - at an error
    \note Uses this method is possible ONLY Since DEVICE was OPENED! (the parameter of stopBits will be ignored otherwise)
*/
bool AbstractSerial::setStopBits(const QString &stopBits)
{
    Q_D(const AbstractSerial);
    int res = d->m_stopBitsMap.key(stopBits);
    if (!res) {
#if defined (ABSTRACTSERIAL_DEBUG)
    qDebug() << "AbstractSerial::setStopBits(const QString &stopBits) \n"
                " -> returned: false because input parameter stopbits:" <<  stopBits << "\n is not defined in QMap. Error!";
#endif
        this->emitStatusString(ENoneSetStopBits);
        return false;
    }
    return this->setStopBits(StopBits(res));
}

/*! \fn QString AbstractSerial::stopBits() const
    \~russian Возвращает количество стоп-бит в виде строки с которым сконфигурировано последовательное устройство.
    \return
    - Количество стоп-бит в виде строки QString
    - 0 - в случае ошибки
    \~english  Returns stop bits as a string which a serial device is configured with.
    \return
    - stop bits as an string QString
    - 0 - in the case of error
*/
QString AbstractSerial::stopBits() const
{
    Q_D(const AbstractSerial);
    if (this->isValid()) {
        AbstractSerial::StopBits r = d->serialEngine->stopBits();
        if (r == AbstractSerial::StopBitsUndefined)
            return QObject::tr("Stop bits undefined");

        QString res = d->m_stopBitsMap.value(r);

        if (res.isEmpty()) {
#if defined (ABSTRACTSERIAL_DEBUG)
    qDebug() << "AbstractSerial::stopBits() -> hash is empty. Error!";
#endif
        }
        return res;
    }
#if defined (ABSTRACTSERIAL_DEBUG)
    qDebug() << "AbstractSerial::stopBits() -> object AbstractSerial is invalid. Error!";
#endif
    return QString();
}

/*! \fn QStringList AbstractSerial::listStopBits() const
    \~russian Возвращает текстовый список всех типов стоп-бит, поддерживаемых классом AbstractSerial.
    \return Список стоп-бит в виде QStringList
    \~english Returns the text list of all of stop bits, supported the class of Abstractserial.
    \return List of stop bits as QStringlist
*/
QStringList AbstractSerial::listStopBits() const
{
    Q_D(const AbstractSerial);
    static const QStringList list = d->m_stopBitsMap.values();
    return list;
}

/*! \fn QMap<AbstractSerial::StopBits, QString> AbstractSerial::stopBitsMap() const
    \~russian Возвращает мапу всех видов стоп бит, поддерживаемых классом AbstractSerial. Используется для заполнения QComboBox.
    \return Список стоп бит как QMap<AbstractSerial::StopBits, QString>
    \~english Returns the map of all stop bit options, supported by the class Abstractserial. Useful for filling up QComboBox.
    \return Map of stop bit options as QMap<AbstractSerial::StopBits, QString>
*/
QMap<AbstractSerial::StopBits, QString> AbstractSerial::stopBitsMap() const
{
    Q_D(const AbstractSerial);
    return d->m_stopBitsMap;
}

/*! \fn bool AbstractSerial::setFlowControl(Flow flow)
    \~russian Устанавливает для последовательного устройства режим управления потоком.
    \param[in] flow - желаемый тип управления потоком последовательного устройства из перечисления.
    \return
    - true - при успехе
    - false - в случае ошибки
    \note Использовать этот метод можно \b ТОЛЬКО \b ПОСЛЕ \b ТОГО \b КАК \b УСТРОЙСТВО \b ОТКРЫЛИ! (иначе параметр \b flow будет игнорироваться)
    \~english Sets a flow control for a serial device
    \param[in] flow - desired flow control of serial device from enumeration.
    \return
    - true - at success
    - false - at an error
    \note Uses this method is possible ONLY Since DEVICE was OPENED! (the parameter of flow will be ignored otherwise)
*/
bool AbstractSerial::setFlowControl(Flow flow)
{
    Q_D(AbstractSerial);
    if (this->isOpen()) {
        bool ret = d->serialEngine->setFlowControl(flow);
        (ret) ? this->emitStatusString(ENoneSetFlow) : this->emitStatusString(ESetFlowControl);
        return ret;
    }
    this->emitStatusString(EDeviceIsNotOpen);
    return false;
}

/*! \fn bool AbstractSerial::setFlowControl(const QString &flow)
    \overload
    \~russian Устанавливает для последовательного устройства режим управления потоком.
    \param[in] flow - желаемый тип управления потоком последовательного устройства в виде строки
    \return
    - true - при успехе
    - false - в случае ошибки
    \note Использовать этот метод можно \b ТОЛЬКО \b ПОСЛЕ \b ТОГО \b КАК \b УСТРОЙСТВО \b ОТКРЫЛИ! (иначе параметр \b flow будет игнорироваться)
    \~english Sets a flow control bits for a serial device
    \param[in] flow -  desired flow of serial device as an string
    \return
    - true - at success
    - false - at an error
    \note Uses this method is possible ONLY Since DEVICE was OPENED! (the parameter of flow will be ignored otherwise)
*/
bool AbstractSerial::setFlowControl(const QString &flow)
{
    Q_D(const AbstractSerial);
    int res = d->m_flowMap.key(flow);
    if (!res) {
#if defined (ABSTRACTSERIAL_DEBUG)
    qDebug() << "AbstractSerial::setFlowControl(const QString &flow) \n"
                " -> returned false because input parameter flow:" <<  flow << "\n is not defined in QMap. Error!";
#endif
        this->emitStatusString(ENoneSetFlow);
        return false;
    }
    return this->setFlowControl(Flow(res));
}

/*! \fn QString AbstractSerial::flowControl() const
    \~russian Возвращает режим управления потоком в виде строки с которым сконфигурировано последовательное устройство.
    \return
    - Режим управления потоком в виде строки QString
    - 0 - в случае ошибки
    \~english  Returns flow control as a string which a serial device is configured with.
    \return
    - flow control as an string QString
    - 0 - in the case of error
*/
QString AbstractSerial::flowControl() const
{
    Q_D(const AbstractSerial);
    if (this->isValid()) {
        AbstractSerial::Flow r = d->serialEngine->flow();
        if (r == AbstractSerial::FlowControlUndefined)
            return QObject::tr("Flow control undefined");

        QString res = d->m_flowMap.value(r);

        if (res.isEmpty()) {
#if defined (ABSTRACTSERIAL_DEBUG)
    qDebug() << "AbstractSerial::flowControl() -> hash is empty. Error!";
#endif
        }
        return res;
    }
#if defined (ABSTRACTSERIAL_DEBUG)
    qDebug() << "AbstractSerial::flowControl() -> object AbstractSerial is invalid. Error!";
#endif
    return QString();
}

/*! \fn QStringList AbstractSerial::listFlowControl() const
    \~russian Возвращает текстовый список всех режимов управления потоком, поддерживаемых классом AbstractSerial.
    \return Список режимов управления потоком в виде QStringList
    \~english Returns the text list of all of flow controls, supported the class of Abstractserial.
    \return List of flow controls as QStringlist
*/
QStringList AbstractSerial::listFlowControl() const
{
    Q_D(const AbstractSerial);
    static const QStringList list = d->m_flowMap.values();
    return list;
}

/*! \fn QMap<AbstractSerial::Flow, QString> AbstractSerial::flowControlMap() const
    \~russian Возвращает мапу всех видов контроля потока, поддерживаемых классом AbstractSerial. Используется для заполнения QComboBox.
    \return Список контроля потока как QMap<AbstractSerial::Flow, QString>
    \~english Returns the map of all flow control options, supported by the class Abstractserial. Useful for filling up QComboBox.
    \return Map of flow control options as QMap<AbstractSerial::Flow, QString>
*/
QMap<AbstractSerial::Flow, QString> AbstractSerial::flowControlMap() const
{
    Q_D(const AbstractSerial);
    return d->m_flowMap;
 }

/*! \fn bool AbstractSerial::setCharIntervalTimeout(int msecs)
    \~russian Устанавливает время ожидания прихода символа в приемный буфер последовательного устройства.
    \param[in] msecs - желаемое время ожидания прихода символа, в мсек
    \return
    - true - при успехе
    - false - в случае ошибки
    \note Использовать этот метод можно \b ТОЛЬКО \b ПОСЛЕ \b ТОГО \b КАК \b УСТРОЙСТВО \b ОТКРЫЛИ! (иначе параметр \b msecs будет игнорироваться)
    \~english Fixes time expectation of arrival of character in the receiving buffer of serial device.
    \param[in] msecs - desired time of expectation of arrival of character, in msek
    \return
    - true - at success
    - false - at an error
    \note Uses this method is possible ONLY Since DEVICE was OPENED! (the parameter of msecs will be ignored otherwise)
*/
bool AbstractSerial::setCharIntervalTimeout(int msecs)
{
    Q_D(AbstractSerial);
    if (this->isOpen()) {
        bool ret = d->serialEngine->setCharIntervalTimeout(msecs);
        (ret) ? this->emitStatusString(ENoneSetCharTimeout) : this->emitStatusString(ESetCharIntervalTimeout);
        return ret;
    }
    this->emitStatusString(EDeviceIsNotOpen);
    return false;
}

/*! \fn int AbstractSerial::charIntervalTimeout() const
    \~russian Возвращает время ожидания прихода символа в приемный буфер последовательного устройства с которым сконфигурировано последовательное устройство.
    \return
    - Время ожидания прихода символа, в мсек
    - 0 - в случае ошибки
    \~english  Returns time of expectation of arrival of character in the receiving buffer of serial device which a serial device is configured with.
    \return
    - Time of expectation of arrival of character, in msek
    - 0 - in the case of error
*/
int AbstractSerial::charIntervalTimeout() const
{
    Q_D(const AbstractSerial);
    return (this->isValid()) ? d->serialEngine->charIntervalTimeout() : 0;
}

/*! \fn bool AbstractSerial::setDtr(bool set)
    \~russian Устанавливает линию DTR в состояное "high" или "low" в зависимости от входного параметра
    \param[in] set - желаемое состояние линии DTR
    \return
    - true - при успехе
    - false - в случае ошибки
    \~english Sets DTR line an state "high" or "low"
    \param[in] set - desired state of line DTR
    \return
    - true - at success
    - false - at an error
*/
bool AbstractSerial::setDtr(bool set)
{
    Q_D(AbstractSerial);
    if (this->isOpen()) {
        bool ret = d->serialEngine->setDtr(set);
        (ret) ? this->emitStatusString(ENoneSetDtr) : this->emitStatusString(ESetDtr);
        return ret;
    }
    this->emitStatusString(EDeviceIsNotOpen);
    return false;
}

/*! \fn bool AbstractSerial::setRts(bool set)
    \~russian Устанавливает линию RTS в состояное "high" или "low" в зависимости от входного параметра
    \param[in] set - желаемое состояние линии RTS
    \return
    - true - при успехе
    - false - в случае ошибки
    \~english Sets DTR line an state "high" or "low"
    \param[in] set - desired state of line RTS
    \return
    - true - at success
    - false - at an error
*/
bool AbstractSerial::setRts(bool set)
{
    Q_D(AbstractSerial);
    if (this->isOpen()) {
        bool ret = d->serialEngine->setRts(set);
        (ret) ? this->emitStatusString(ENoneSetRts) : this->emitStatusString(ESetRts);
        return ret;
    }
    this->emitStatusString(EDeviceIsNotOpen);
    return false;
}

/*! \fn ulong AbstractSerial::lineStatus()
    \~russian Возвращает статусы линий CTS, DSR, DCD, RI, RTS, DTR, ST, SR.
    Метод возвращает закодированные значения статусов линий побитовым ИЛИ.
    \return
    - true - при успехе
    - false - в случае ошибки
    \~english Returned the status lines CTS, DSR, DCD, RI, RTS, DTR, ST, SR.
    The method returns an encoded value status lines bitwise OR.
    \return
    - true - at success
    - false - at an error
*/
ulong AbstractSerial::lineStatus()
{
    Q_D(AbstractSerial);
    if (this->isOpen()) {
        bool ret = d->serialEngine->lineStatus();
        if (!ret)
            this->emitStatusString(ELineStatus);
        return ret;
    }
    this->emitStatusString(EDeviceIsNotOpen);
    return false;
}

/*! \fn bool AbstractSerial::sendBreak(int duration)
    \~russian Передает непрерывный поток нулевых битов в течение указанного промежутка времени,
    если терминал использует асинхронную последовательность передачи данных.
    \param[in] duration - промежутокк времени, в мсек:
    - если duration равно нулю, то нулевые биты передаются, по меньшей мере, в течение 0.25 секунд, но не более 0.5 секунды
    - если duration не равно нулю, то нулевые биты посылаются в течении некоторого периода времени, зависящего от реализации
    \return
    - true - при успехе
    - false - в случае ошибки
    \~english If the terminal is using asynchronous serial data transmission, shall cause transmission of a
    continuous stream of zero-valued bits for a specific duration.
    \param[in] duration - period of time, in msecs:
    - if duration is 0, it shall cause transmission of zero-valued bits for at least 0.25 seconds, and not more than 0.5 seconds
    - if duration is not 0, it shall send zero-valued bits for an implementation-defined period of time
    \return
    - true - at success
    - false - at an error
*/
bool AbstractSerial::sendBreak(int duration)
{
    Q_D(AbstractSerial);
    if (this->isOpen()) {
        if (duration < 0)
            duration = 0;
        bool ret = d->serialEngine->sendBreak(duration);
        if (!ret)
            this->emitStatusString(ESendBreak);
        return ret;
    }
    this->emitStatusString(EDeviceIsNotOpen);
    return false;
}

/*! \fn bool AbstractSerial::setBreak(bool set)
    \~russian Включает/отключает сигнал разрыва (начинает/прекращает отправлять нулевые биты).
    \param[in] set - флаг включения/отключения передачи нулевых битов:
    - если set=true - включить передачу
    - если set=false - отключить передачу
    \return
    - true - при успехе
    - false - в случае ошибки
    \~english Enables/disables the signal break (start/stop sending zero bits).
    \param[in] set - flag on/off transmission zeros:
    - if set=true - enable the transmission
    - if set=false - disable the transmission
    \return
    - true - at success
    - false - at an error
*/
bool AbstractSerial::setBreak(bool set)
{
    Q_D(AbstractSerial);
    if (this->isOpen()) {
        bool ret = d->serialEngine->setBreak(set);
        if (!ret)
            this->emitStatusString(ESetBreak);
        return ret;
    }
    this->emitStatusString(EDeviceIsNotOpen);
    return false;
}

/*! \fn bool AbstractSerial::flush()
    \~russian Очищает буферы последовательного устройства.
    \return
    - true - при успехе
    - false - в случае ошибки
    \~english Clears the buffers of serial device.
    \return
    - true - at success
    - false - at an error
*/
bool AbstractSerial::flush()
{
    Q_D(AbstractSerial);
    if (this->isOpen()) {
        bool ret = d->serialEngine->flush();
        if (!ret)
            this->emitStatusString(EFlush);
        return ret;
    }
    this->emitStatusString(EDeviceIsNotOpen);
    return false;
}

/*! \fn bool AbstractSerial::reset()
    \~russian Сбрасывает буферы последовательного устройства.
    \return
    - true - при успехе
    - false - в случае ошибки
    \~english Reset the buffers of serial device.
    \return
    - true - at success
    - false - at an error
*/
bool AbstractSerial::reset()
{
    Q_D(AbstractSerial);
    if (this->isOpen()) {
        bool ret = d->serialEngine->reset();
        d->emittedReadyRead = false;
        return ret;
    }
    this->emitStatusString(EDeviceIsNotOpen);
    return false;
}

/*! \fn qint64 AbstractSerial::bytesAvailable() const
    \~russian Возвращает количество байт готовых для чтения,
    которые находятся во входном буфере последовательного устройства
    \return
    - n - количество байт при успехе
    - -1 - в случае ошибки
    \~english Returns the number of bytes ready to read,
    who are in the input buffer serial device
    \return
    - n - is an amount byte at success
    - -1 - in the case of error
*/
qint64 AbstractSerial::bytesAvailable() const
{
    Q_D(const AbstractSerial);
    if (this->isOpen()) {
        return d->serialEngine->bytesAvailable();
    }
    return (qint64)-1;
}

/*
   Returns the difference between msecs and elapsed. If msecs is -1,
   however, -1 is returned.
*/
static int qt_timeout_value(int msecs, int elapsed)
{
    if (msecs == -1)
        return msecs;
    int timeout = msecs - elapsed;
    return (timeout < 0) ? 0 : timeout;
}

/*! \fn bool AbstractSerial::waitForReadyRead(int msecs)
    \~russian Ожидает прихода во входной буфер последовательного устройства хотя-бы одного байта данных в течении определенного времени.
    При успешном выполнении излучается сигнал readyRead().
    \param[in] msecs - время в течении которого ожидается приход символа, в мсек
    \return
    - true - при успехе
    - false - в случае ошибки или при таймауте
    \~english Expects arrival in the entrance buffer of serial device of even one byte of data in the flow of set time.
    The signal of readyRead() emitted at successful implementation().
    \param[in] msecs - time in the flow of which is expected arrival of character, in msek
    \return
    - true - at success
    - false - in the case of error or at a timeout
*/
bool AbstractSerial::waitForReadyRead(int msecs)
{
    Q_D(AbstractSerial);
    if (!this->isOpen()) {
        this->emitStatusString(EDeviceIsNotOpen);
        return false;
    }

    QTime stopWatch;
    stopWatch.start();

    forever {
        bool readyToRead = false;
        bool readyToWrite = false;
        if (!d->serialEngine->waitForReadOrWrite(&readyToRead, &readyToWrite, true, false,
                                                    qt_timeout_value(msecs, stopWatch.elapsed()))) {
            this->emitStatusString(EWaitReadyReadTimeout);
            return false;
        }
        if (readyToRead) {
            d->emittedReadyRead = false;
            d->_q_canReadNotification();
            return true;
        }
    }
}

/*! \fn bool AbstractSerial::waitForBytesWritten(int msecs)
    \~russian Ожидает ухода из выходного буфера последовательного устройства последнего символа в течении определенного времени.
    \note В данной версии библиотеки этот метод пока не реализован.
    \~english Expects a care from the output buffer of serial device of the last character in the flow of set time.
    \note In this version of library this method while is not implement.
*/
bool AbstractSerial::waitForBytesWritten(int msecs)
{
    Q_UNUSED(msecs)
    return false;
}

/*! \fn void AbstractSerial::enableEmitStatus(bool enable)
    \~russian Включает или отключает флаг испускания сигнала signalStatus()
    \param[in] enable - желаемое значение флага
    \note если:
    - enable = true - то это значит, что включить испускание сигнала signalStatus()
    - enable = false - то это значит, что отключить испускание сигнала signalStatus()
    \~english On or Off the flag of emitting of signal of signalStatus()
    \param[in] enable - desired value of flag
    \note if:
    - enable = true -  it means that to On emitting of signal of signalStatus()
    - enable = false -  it means that to Off emitting of signal of signalStatus()
*/
void AbstractSerial::enableEmitStatus(bool enable)
{
    Q_D(AbstractSerial);
    d->emittedStatus = enable;
}

/*! \fn qint64 AbstractSerial::read(char *data, qint64 maxSize)
    \~russian Читает из последовательного устройства maxSize байт данных в буфер data
    \param[out] data - указатель на массив (буфер) в который будут прочитаны данные
    \param[in] maxSize - длина данных, которые хотим прочитать
    \return
    - n - количество реально прочитанных байт
    - -1 - в случае ошибки
    \note количество реально прочитанных байт может быть меньше чем maxSize или может быть равно нулю
    \~english Reads at most maxSize bytes from the ыукшфд device into data
    \param[in] data - pointer to an array (buffer) which will read data
    \param[in] maxSize - length of data that we want to read
    \return
    - n - number of bytes actually read
    - -1 - in case of error
    \note the number of bytes actually read may be less than maxSize, or may be zero
*/
qint64 AbstractSerial::read(char *data, qint64 maxSize)
{
    Q_D(AbstractSerial);
    qint64 ret = -1;
    if (!this->isOpen()) {
        this->emitStatusString(EDeviceIsNotOpen);
        return ret;
    }

    if (d->serialEngine->openMode() == AbstractSerial::WriteOnly) {
        return ret;
    }

    ret = d->serialEngine->read(data, maxSize);
    if (ret < 0) {
        this->emitStatusString(EReadDataIO);
    }

    d->emittedReadyRead = false;

    return ret;
}

/*! \fn QByteArray AbstractSerial::read(qint64 maxSize)
    \overload
    \~russian Читает из последовательного устройства maxSize байт, и
    возвращает прочитанные данные в виде QByteArray
    \param[in] maxSize - длина данных, которые хотим прочитать
    \return массив прочитанных данных в виде QByteArray
    \note Эта функция не может сообщать об ошибках чтения. Возвращение пустого
    QByteArray() может означать, что либо нет данных доступных для чтения в настоящее время,
    или то что произошла ошибка.
    \~english Reads from the serial device maxSize bytes, and returns the
    data read as a QByteArray.
    \param[in] maxSize - length of data that we want to read
    \return read array data as QByteArray
    \note This function has no way of reporting errors; returning an empty
    QByteArray() can mean either that no data was currently available
    for reading, or that an error occurred.
*/
QByteArray AbstractSerial::read(qint64 maxSize)
{
    Q_D(AbstractSerial);
    QByteArray result;
    if (!this->isOpen()) {
        this->emitStatusString(EDeviceIsNotOpen);
        return result;
    }

    if (d->serialEngine->openMode() == AbstractSerial::WriteOnly) {
        return result;
    }

    /* TODO: What else is there to limit maxSize (qint64) to the size of INT_MAX (see limits.h)?
       ...
       ...
    */

    for (;;) {
        qint64 bytesToRead = qMin<qint64>(ABSTRACTSERIAL_READ_CHUNK_SIZE, maxSize);
        qint64 ret = d->serialEngine->read(d->rxChunkBuffer, bytesToRead);

        if (ret < 0) {
            this->emitStatusString(EReadDataIO);
            result.clear();
            break;
        }
        if (ret > 0) {
            result.append(d->rxChunkBuffer, int(ret));
            if (ret < bytesToRead)
                break;
            maxSize -= ret;
            if (maxSize == 0)
                break;
        }
        else
            break;
    }

    d->emittedReadyRead = false;

    return result;
}

/*! \fn QByteArray AbstractSerial::readAll()
    \overload
    \~russian Читает из последовательного устройства все доступные данные и
    возвращает их как массив QByteArray
    \return массив прочитанных данных в виде QByteArray
    \note Эта функция не может сообщать об ошибках чтения. Возвращение пустого
    QByteArray() может означать, что либо нет данных доступных для чтения в настоящее время,
    или то что произошла ошибка.
    \~english Reads from the serial device all available data
    and returns it as a QByteArray.
    \return read array data as QByteArray
    \note This function has no way of reporting errors; returning an empty
    QByteArray() can mean either that no data was currently available
    for reading, or that an error occurred.
*/
QByteArray AbstractSerial::readAll()
{
    Q_D(AbstractSerial);
    QByteArray result;
    if (!this->isOpen()) {
        this->emitStatusString(EDeviceIsNotOpen);
        return result;
    }

    if (d->serialEngine->openMode() == AbstractSerial::WriteOnly) {
        return result;
    }

    // Size is unknown, read incrementally.
    for (;;) {
        qint64 ret = d->serialEngine->read(d->rxChunkBuffer, ABSTRACTSERIAL_READ_CHUNK_SIZE);

        if (ret < 0) {
            this->emitStatusString(EReadDataIO);
            result.clear();
            break;
        }
        if (ret > 0) {
            result.append(d->rxChunkBuffer, int(ret));
        }
        else
            break;
    }

    d->emittedReadyRead = false;

    return result;
}

/*! \fn qint64 AbstractSerial::write(const char *data, qint64 maxSize)
    \~russian Записывает в последовательное устройство maxSize байт данных из буфера data
    \param[in] data - указатель на массив (буфер) из который будут прочитаны данные
    \param[in] maxSize - длина данных, которые хотим записать
    \return
    - n - количество реально записанных байт
    - -1 - в случае ошибки
    \~english Writes at most maxSize bytes of data from data to the serial device.
    \param[in] data - pointer to an array (buffer) from which data will be write
    \param[in] maxSize - length of the data that we want to write
    \return
    - n - number of bytes actually written
    - -1 - in case of error.
*/
qint64 AbstractSerial::write(const char *data, qint64 maxSize)
{
    Q_D(AbstractSerial);
    qint64 ret = -1;
    if (!this->isOpen()) {
        this->emitStatusString(EDeviceIsNotOpen);
        return ret;
    }

    if (d->serialEngine->openMode() == AbstractSerial::ReadOnly) {
        return ret;
    }

    ret = d->serialEngine->write(data, maxSize);
    if (ret < 0)
        this->emitStatusString(EWriteDataIO);
    else
        emit this->bytesWritten(ret);
    return ret;
}

/*! \fn qint64 AbstractSerial::qint64 write(const char *data)
    \overload
    \~russian Записывает в последовательное устройство данные в виде нуль-терминированной строки 8-битных символов.\n
    Эквивалент:
    \code
        ...
        AbstractSerial::write(data, qstrlen(data));
        ...
    \endcode
    \param[in] data - указатель на массив (буфер/строку 8-битных символов) из который будут записаны данные
    \return
    - n - количество реально записанных байт
    - -1 - в случае ошибки
    \~english Writes data from a zero-terminated string of 8-bit characters to the
    serial device.\n
    Equivalent:
    \code
        ...
        AbstractSerial::write(data, qstrlen(data));
        ...
    \endcode
    \param[in] data - pointer to an array (buffer/string 8-bit char) from which data will be write
    \return
    - n - number of bytes actually written
    - -1 - in case of error.
*/
qint64 AbstractSerial::write(const char *data)
{
    Q_D(AbstractSerial);
    qint64 ret = -1;
    if (!this->isOpen()) {
        this->emitStatusString(EDeviceIsNotOpen);
        return ret;
    }

    if (d->serialEngine->openMode() == AbstractSerial::ReadOnly) {
        return ret;
    }

    ret = d->serialEngine->write(data, qstrlen(data));
    if (ret < 0)
        this->emitStatusString(EWriteDataIO);
    else
        emit this->bytesWritten(ret);
    return ret;
}

/*! \fn qint64 AbstractSerial::write(const QByteArray &byteArray)
    \overload
    \~russian Записывает в последовательное устройство данные содержащиеся в byteArray
    \param[in] byteArray - массив данных которые нужно записать
    \note размер данных ограничен только размером QByteArray
    \return
    - n - количество реально записанных байт
    - -1 - в случае ошибки
    \~english Writes the content of byteArray to the serial device.
    \param[in] byteArray - array of data that you want to write
    \note data size is limited only by the size QByteArray
    \return
    - n - number of bytes actually written
    - -1 - in case of error.
*/
qint64 AbstractSerial::write(const QByteArray &byteArray)
{
    Q_D(AbstractSerial);
    qint64 ret = -1;
    if (!this->isOpen()) {
        this->emitStatusString(EDeviceIsNotOpen);
        return ret;
    }

    if (d->serialEngine->openMode() == AbstractSerial::ReadOnly) {
        return ret;
    }

    ret = d->serialEngine->write(byteArray.data(), byteArray.size());
    if (ret < 0)
        this->emitStatusString(EWriteDataIO);
    else
        emit this->bytesWritten(ret);
    return ret;
}

/*! \fn void AbstractSerial::emitStatusString(Status status)
    \~russian Испускает сигнал signalStatus()
    \param[in] status - тип статуса
    \~english  Emits the signal of signalStatus()
    \param[in] status - type of status
*/
void AbstractSerial::emitStatusString(Status status)
{
    Q_D(const AbstractSerial);
    if (this->canEmitStatusString())
        emit this->signalStatus(d->statusToString(status), QDateTime::currentDateTime());
}

/*! \fn bool AbstractSerial::canEmitStatusString() const
    \~russian Возвращает значение флага, который определяет испускать или нет сигнал signalStatus()
    \return Значение флага
    \note если:
    - true - то это значит, что испускание сигнала signalStatus() включено
    - false - то это значит, что испускание сигнала signalStatus() отключено
    \~english Returns the value of flag which determines to emit or not signal of signalStatus()
    \return flag value
    \note if:
    - true - it means that emitting of signal of signalStatus() it is On
    - false - it means that emitting of signal of signalStatus() it is Off.
*/
bool AbstractSerial::canEmitStatusString() const
{
    Q_D(const AbstractSerial);
    return d->emittedStatus;
}

/*! \fn bool AbstractSerial::isValid() const
*/
bool AbstractSerial::isValid() const
{
    Q_D(const AbstractSerial);
    return (d->serialEngine != 0);
}

/*! \fn void AbstractSerial::readyRead()
    \~russian Этот сигнал испускается один раз при приходе в приемный буфер последовательного устройства
    хотя бы одного байта, т.е. если вы сделали "connect" к этому сигналу после того как
    в приемный буфер последовательного устройства пришли данные - то ваш слот не отработает, т.к.
    сигнал уже излучился. Поэтому "connect" желательно делать перед тем как вы открываете порт!\n
    Для того чтобы сигнал снова испустился необходимо прочесть данные ( read() ) из последовательного устройства
    или сбросить(очистить) буфер последовательного устройства ( reset() ).\n
    В процессе чтения данных этот сигнал не испускается (блокируется), это сделано для того чтобы
    процесс чтения завершился без сбоев. После того как данные прочитали - сигнал разблокируется.

    \~english This signal is emitted once at arrival in the receiving buffer serial device
    at least one byte, ie if you did "connect" to this signal after
    in the receive buffer serial device data came - then your slot is not developed, because
    signal is rejected. Therefore, "connect" it is desirable to do before you open a port! \n
    In order to signal again gave up the need to read data from serial device ( read() )
    or reset (clear) buffer serial device ( reset() ). \n
    In the process of reading data, this signal is not emitted (blocked), it is done to
    reading process was completed smoothly. Once the data is read - the signal is unblocked.

    \~
    \note \~russian Применять этот сигнал можно в приложении например таким образом:
    \note \~english Use This signal can be in the application for example as follows:
    \~

    \code
    //constructor
    MyClass::MyClass(QObject *parent)
    {
        ...
        port = new AbstractSerial(this);
        ...
        //here set device name
        ...
        //< here to do "connect" is recommended (before the opening)
        connect( port, SIGNAL(readyRead()), this, SLOT(readDataSlot()) );
        ...

        //here open port
        port->open(mode);

        //< here to do "connect" is not recommended (after the opening)
        ....
        //here configure port
        ...
     }

     //slot
     void MyClass::readDataSlot()
     {
        ...
        QByteArray data = port->readAll();
        ...
     }
    \endcode
*/

/*! \fn void AbstractSerial::bytesWritten(qint64 bytes);
    \~russian Этот сигнал испускается каджый раз при успешной записи данных в последовательное устройство.
    \param[out] bytes - кол-во байт, которые записали реально в последовательное устройство

    \~english This signal is emitted every time you successfully write data to the serial device.
    \param[out] bytes - number of bytes that are actually write to the serial device
*/

/*! \fn void AbstractSerial::signalStatus(const QString &status, QDateTime current)
    \~russian Этот сигнал автоматически испускается всякий раз при изменении статуса объекта AbstractSerial. \n
    Изменение статуса может быть вызвано следующими причинами:
    - при ошибках конфигурировании устройства
    - при его открытии/закрытии
    - при ошибках чтения/записи данных и т.п.
    \param[out] status - текущий статус последовательного устройства
    \param[out] current - текущее значение даты/времени произошедшей смены статуса
    \~english This signal is emitted automatically every time you change the status of the object AbstractSerial.\n
    The change of status can be caused the followings reasons:
    - at errors configuring of device
    - at his opening/closing
    - at the errors of reading/writting of data etc.

    \~
    \note \~russian Применять этот сигнал можно в приложении например таким образом:
    \note \~english Use This signal can be in the application for example as follows:
    \~

    \code
    //constructor
    MyClass::MyClass(QObject *parent)
    {
        ...
        port = new AbstractSerial(this);
        ...
        connect( port, SIGNAL(signalStatus(const QString &, QDateTime)), this, SLOT(viewStateSlot(QString &, QDateTime)) );
        ...
        //here set device name
        ...
        //here open port
        ....
        //here configure port
        ...
     }

     //slot
     void MyClass::viewStateSlot(QString &stateMsg, QDateTime dt)
     {
        ...
        qDebug() << "State: " << stateMsg << ", in time: " << dt.time().toString();
        ...
     }
    \endcode
*/
