#pragma once

#include <iostream>
#include <ostream>
#include <string>
#include "ManagedBuffer.h"
#include <sstream>
#include <iomanip>
#include <limits>
#include "LinkedList.h"

template <class _T> 
class basic_Matrix2D {
public:

	////Whether this matrix is built temporary
	/// in this case the matrix is not duplicated,
	/// the data is reused
	bool Temporary;

	////Creates a new matrix from the size of rows and columns.
	basic_Matrix2D(int rows, int cols) : data(*(new ManagedBuffer<_T>())) {
		this->cols=cols;
		this->rows=rows;
		data.Resize(cols*rows);
		Temporary=false;
	}

        basic_Matrix2D(const basic_Matrix2D &source) : data(source.data) {
            *this=source;
        }

	////Creates a new matrix of the given size.
	basic_Matrix2D(Size2D size) : data(*(new ManagedBuffer<_T>())) {
		cols=size.Width;
		rows=size.Height;
		data.Resize(cols*rows);
		Temporary=false;
	}
	////Creates an empty matrix without a data buffer.
	basic_Matrix2D() : data(*(new ManagedBuffer<_T>())) {
		cols=0;
		rows=0;
		Temporary=false;
	}

	////Initializes this matrix from another one, this does not copy a matrix
	/// instead it uses same data buffer, linking these two matrices.
	basic_Matrix2D(basic_Matrix2D &source) : data(source.data) {
		this->cols=source.cols;
		this->rows=source.rows;
                this->data=++data;
		Temporary=false;
	}
	////Removes a reference count from the data buffer and destroys this matrix.
	~basic_Matrix2D() {
                --data;
	}

	////Fills in the matrix with given data.
	/// be careful about data type. They should be 
	/// same as template argument. ie when the type
	/// is double, data should be specified as 5.0
	/// The number of arguments given to this function
	/// should be the size of the matrix. eg 3x3 matrix
	/// should take 9 arguments.
	void SetData(_T value, ...) {
		int i;

		_T *v=&value;

		for(i=0;i<rows*cols;i++) {
			data[i]=*v;
			v++;
		}
	}

	////Returns the value in the given matrix location
	/// specified with row and col parameters.
	/// the result of this function is a reference, so
	/// a value can be assign to it. eg A.Get(1,2)=5;
	/// In the debug mode, this function does bound checking. 
	///-MatrixException:MatrixExceptions.ME_OutOfBounds
	_T &Get(int row, int col) throw(MathException) {
#ifndef  MATH_NOERRORCHECK
		if(row>=rows || row<0 || col<0 || col>=cols) {
			_T t;
			throw MathException(ME_OutOfBounds);
			return t;
		}
#endif

		return data[col+row*cols];
	}

	////Returns the value in the given matrix location
	/// specified with row and col parameters.
	/// the result of this function is a reference, so
	/// a value can be assign to it. eg A(1,2)=5;
	/// In the debug mode, this function does bound checking. 
	///-MatrixException:MatrixExceptions.ME_OutOfBounds
	_T &operator ()(int row, int col) throw(MathException) {
#ifndef  MATH_NOERRORCHECK
		if(row>=rows || row<0 || col<0 || col>=cols) {
			_T t;
			throw MathException(ME_OutOfBounds);
			return t;
		}
#endif

		return data[col+row*cols];
	}

	////Assigns the given value to given matrix location
	///specified with row and col parameters.
	///-MatrixException:MatrixExceptions.ME_OutOfBounds
	void Set(int row, int col, _T value) throw(MathException) {
#ifndef  MATH_NOERRORCHECK
		_T t;
		if(row>=cols || row<0 || col<0 || col>=rows) {
			_T t;
			throw MathException(ME_OutOfBounds);
			return t;
		}
#endif

		data[row+col*cols]=value;
	}
	
	
	////Changes the size of matrix with the given
	/// rows and cols parameters. This function
	/// preserves all data however the row and column 
	/// positions might shift.
	void Resize(int cols, int rows) throw() {
		data.Resize(cols*rows);
	}
	
	////Returns the size of matrix.
	Size2D GetSize() throw() { return Size2D(cols, rows); }
	
	////Returns the column number of matrix.
	int GetCols() throw() { return cols; }
	
	////Returns the row number of matrix.
	int GetRows() throw() { return rows; }
	
	
	////Returns the data array.
	_T *GetData() throw() { return data; }

	////Checks whether the matrix is square matrix.
	bool isSquare() throw() { return rows==cols; }
	
	////Calculates the median of a given row.
	///-MatrixException:MatrixExceptions.ME_InexistantRow
	_T RowMedian(int row){
#ifndef  MATH_NOERRORCHECK
		if(row>=rows){
			throw MathException(ME_InexistantRow);
		}
#endif
		int col;
		LinkedList<int> list;
		for(col=0;col<cols;col++) {
			int *colindex=new int[1];

			*colindex=col;
			list.AddItem(colindex, Get(row,col));
		}
		int a=*list.OrderedItemAt(cols/2);
		_T median=data[a+row*cols];		
	
		list.Destroy();
	
		return median; 
		
	}

	////Checks the given matrix whether is equal to current one.
	///-MatrixException:MatrixExceptions.ME_MatrixSizeNotEqual
	bool Compare(basic_Matrix2D &mat){
#ifndef  MATH_NOERRORCHECK
		if(mat.rows!=rows || mat.cols!=cols){
			throw MathException(ME_MatrixSizeNotEqual);
		}
#endif

		_T v=std::numeric_limits<_T>::epsilon();
		if(v<1)
			v*=5;

		int row,col;
		for(row=0;row<rows;row++){
			for(col=0;col<cols;col++){
				_T sub=mat[col+row*cols]-data[col+row*cols];
				if(sub>v || sub<-v)
					return false;
		

			}
		}

		return true;
	}
	
	////Sorts the  rows of the matrix depending on the value of the given column.
	///-MatrixException:MatrixExceptions.ME_InexistantRow
	basic_Matrix2D &SortRows(int col) {
#ifndef  MATH_NOERRORCHECK
		if(col>=cols){
			throw MathException(ME_InexistantColumn);
		}
#endif
		int row;
		LinkedList<int> list;
		for(row=0;row<rows;row++) {
			int *rowindex=new int[1];

			*rowindex=row;
			list.AddItem(rowindex, Get(row, col));
		}

		LinkedListOrderedIterator<int> it=list;
		int *rowindex;
		row=0;
		while(rowindex=it) {
			SwapRows(*rowindex, row);
			*list[row]=*rowindex;

			row++;
		}
		
		list.Destroy();

		return *this;
	}
	////Removes the specified column of the matrix.
	/// The function returns the current matrix.
	///-MatrixException:MatrixExceptions.ME_InexistantColumn	
	basic_Matrix2D &RemoveCol(int Col) {
#ifndef  MATH_NOERRORCHECK
                if(Col>=cols){
			throw MathException(ME_InexistantColumn);
		}
#endif
		for(int row=0;row<rows;row++) {
			for(int col=0,c=0;col<cols-1;col++,c++) {
				if(c==Col)
					c++;
				data[row*(cols-1)+col]=Get(row,c);
			}
		}

		cols--;
		//data.Resize(cols*rows);
		return *this;
	}
	////Removes the specified row of the matrix.
	/// The function returns the current matrix.
	///-MatrixException:MatrixExceptions.ME_InexistantRow
	basic_Matrix2D &RemoveRow(int Row){
#ifndef  MATH_NOERRORCHECK
                if(Row>=rows){
			throw MathException(ME_InexistantRow);
		}
#endif
		for(int row=0,r=0;row<rows-1;row++,r++) {
			if(row==Row)
				r++;
			for(int col=0;col<cols;col++) {
				Get(row, col)=Get(r,col);
			}
		}

		rows--;
		//data.Resize(cols*rows);
		return *this;
	}


	////Checks the given matrix whether is equal to this one.
	///-MatrixException:MatrixExceptions.ME_MatrixSizeNotEqual
	bool operator ==(basic_Matrix2D &mat){ 
#ifndef  MATH_NOERRORCHECK
		if(mat.rows!=rows || mat.cols!=cols){
			throw MathException(ME_MatrixSizeNotEqual);
		}
#endif

		return Compare(mat);
	}

	////Copies the current matrix to another matrix where
	/// the location is determined.
	///-MatrixException:MatrixExceptions.ME_SizeNotEqualOrGreat
	basic_Matrix2D CopyTo(basic_Matrix2D &mat, int row, int col){
#ifndef  MATH_NOERRORCHECK
		if(rows+row>mat.rows){
			throw MathException(ME_SizeNotEqualOrGreat);
		}
		if(cols+col>mat.cols){
			throw MathException(ME_SizeNotEqualOrGreat);
		}
#endif
		int index=0;
		for(int i=row;i<rows+row;i++){
			for(int j=col;j<cols+col;j++){
				mat[j+i*cols]=data[index];
				index++;
			}
		}
		
		return mat;
		
	}


	////Initializes this matrix from the right hand matrix by copying that one
        basic_Matrix2D operator =(basic_Matrix2D matrix) {
                --data;
		cols=matrix.cols;
		rows=matrix.rows;
		if(matrix.Temporary)
			data=matrix.data;
		else {
			int i;
			data.Resize(rows*cols);
			for(i=0;i<rows*cols;i++)
				data[i]=matrix[i];
		}
                ++data;

		return *this;
	}

	////Find and returns the maximum number in the matrix.
	///-MatrixException:MatrixExceptions.ME_MatrixEmpty
	_T Max() {
#ifndef MATH_NOERRORCHECK
		if(data.GetSize()==0){
			throw MathException(ME_MatrixEmpty);
		}
#endif

		_T max=data[0];
		int index;
		for(index=0;index<rows*cols;index++){
			if(data[index]>max)
				max=data[index];
		}

		return max;
	}

	////Find and returns the index of maximum number in the matrix.
	///-MatrixException:MatrixExceptions.ME_MatrixEmpty
	int MaxIndex() {
#ifndef MATH_NOERRORCHECK
		if(data.GetSize()==0){
			throw MathException(ME_MatrixEmpty);
		}
#endif

		_T max=data[0];
		int index=0;
		for(int i=0;i<rows*cols;i++){
			if(data[i]>max){
				max=data[i];
				index=i;
			}
		}

		return index;
	}

	////Find and returns the minimum number in the matrix.
	///-MatrixException:MatrixExceptions.ME_MatrixEmpty
	_T Min(){
#ifndef MATH_NOERRORCHECK
		if(data.GetSize()==0){
			throw MathException(ME_MatrixEmpty);
		}
#endif


		_T min=data[0];
		int index;
		for(index=0;index<rows*cols;index++){
			if(data[index]<min)
				min=data[index];
		}

		return min;
	}

	////Find and returns the index of minimum number in the matrix.
	///-MatrixException:MatrixExceptions.ME_MatrixEmpty
	int MinIndex(){
#ifndef MATH_NOERRORCHECK
		if(data.GetSize()==0){
			throw MathException(ME_MatrixEmpty);
		}
#endif

		_T min=data[0];
		int index=0;
		for(int i=0;i<rows*cols;i++){
			if(data[i]<min){
				min=data[i];
				index=i;
			}
		}

		return index;
	}
	////Find and returns the maximum number in the specified column of matrix.
	///-MatrixException:MatrixExceptions.ME_MatrixEmpty
	///-MatrixException:MatrixExceptions.ME_InexistantColumn	
	_T ColMax(int col){
#ifndef MATH_NOERRORCHECK
		if(data.GetSize()==0){
			throw MathException(ME_MatrixEmpty);
		}
		if(col>=cols){
			throw MathException(ME_InexistantColumn);
		}
#endif

		_T max=data[0];
		for(int i=col;i<rows*cols;i+=cols){
			if(data[i]>max){
				max=data[i];
			}
		}

		return max;
	}

	////Find and returns the index of maximum number in the specified column of matrix.
	///-MatrixException:MatrixExceptions.ME_MatrixEmpty
	///-MatrixException:MatrixExceptions.ME_InexistantColumn	
	int ColMaxIndex(int col){
		#ifndef MATH_NOERRORCHECK
		if(data.GetSize()==0){
			throw MathException(ME_MatrixEmpty);
		}
		if(col>=cols){
			throw MathException(ME_InexistantColumn);
		}
#endif

		_T max=data[0];
		int index=0;
		for(int i=col;i<rows*cols;i+=cols){
			if(data[i]>max){
				max=data[i];
				index=i;
			}
		}

		return index/cols;
	}	
	
	////Find and returns the minimum number in the specified column of matrix.
	///-MatrixException:MatrixExceptions.ME_MatrixEmpty
	///-MatrixException:MatrixExceptions.ME_InexistantColumn	
	_T ColMin(int col){
#ifndef MATH_NOERRORCHECK
		if(data.GetSize()==0){
			throw MathException(ME_MatrixEmpty);
		}
		if(col>=cols){
			throw MathException(ME_InexistantColumn);
		}
#endif

		_T min=data[0];
		for(int i=col;i<rows*cols;i+=cols){
			if(data[i]<min){
				min=data[i];
			}
		}

		return min;
	}
	
	////Find and returns the index of minimum number in the specified column of matrix.
	///-MatrixException:MatrixExceptions.ME_MatrixEmpty
	///-MatrixException:MatrixExceptions.ME_InexistantColumn	
	int ColMinIndex(int col){
		#ifndef MATH_NOERRORCHECK
		if(data.GetSize()==0){
			throw MathException(ME_MatrixEmpty);
		}
		if(col>=cols){
			throw MathException(ME_InexistantColumn);
		}
#endif

		_T min=data[0];
		int index=0;
		for(int i=col;i<rows*cols;i+=cols){
			if(data[i]<min){
				min=data[i];
				index=i;
			}
		}

		return index/cols;
	}

	////Find and returns the maximum number in the given row of matrix.
	///-MatrixException:MatrixExceptions.ME_MatrixEmpty
	///-MatrixException:MatrixExceptions.ME_InexistantRow	
	_T RowMax(int row){
#ifndef MATH_NOERRORCHECK
		if(data.GetSize()==0){
			throw MathException(ME_MatrixEmpty);
		}
		if(row>=rows){
			throw MathException(ME_InexistantRow);
		}
#endif

		_T max=data[0];
		for(int i=row*cols;i<rows*cols-cols;i++){
			if(data[i]>max){
				max=data[i];
			}
		}

		return max;
	}
	////Find and returns the index of maximum number in the given row of matrix.
	///-MatrixException:MatrixExceptions.ME_MatrixEmpty
	///-MatrixException:MatrixExceptions.ME_InexistantRow	
	int RowMaxIndex(int row){
		#ifndef MATH_NOERRORCHECK
		if(data.GetSize()==0){
			throw MathException(ME_MatrixEmpty);
		}
		if(row>=rows){
			throw MathException(ME_InexistantRow);
		}
#endif

		_T max=data[0];
		int index=0;
		for(int i=row*cols;i<rows*cols-cols;i++){
			if(data[i]>max){
				max=data[i];
				index=i;
			}
		}

		return index%cols;
	}
	////Find and returns the minimum number in the specified row of matrix.
	///-MatrixException:MatrixExceptions.ME_MatrixEmpty
	///-MatrixException:MatrixExceptions.ME_InexistantRow	
	_T RowMin(int row){
		#ifndef MATH_NOERRORCHECK
		if(data.GetSize()==0){
			throw MathException(ME_MatrixEmpty);
		}
		if(row>=rows){
			throw MathException(ME_InexistantRow);
		}
#endif

		_T min=data[0];
		for(int i=row*cols;i<row*cols+cols;i++){
			if(data[i]<min){
				min=data[i];
			}
		}

		return min;
	}

	////Find and returns the index of minimum number in the specified row of matrix.
	///-MatrixException:MatrixExceptions.ME_MatrixEmpty
	///-MatrixException:MatrixExceptions.ME_InexistantRow	
	int RowMinIndex(int row){
#ifndef MATH_NOERRORCHECK
		if(data.GetSize()==0){
			throw MathException(ME_MatrixEmpty);
		}
		if(row>=rows){
			throw MathException(ME_InexistantRow);
		}
#endif

		_T min=data[0];
		int index=0;
		for(int i=row*cols;i<row*cols+cols;i++){
			if(data[i]<min){
				min=data[i];
				index=i;
			}
		}

		return index%cols;
	}


	////Creates a normalized version of this matrix.
	basic_Matrix2D Normalize(){ return (*this)/Sum(); }
	
	////Returns the average of the matrix values.
	_T Average() { return Sum()/(rows*cols); }

	////Returns sum of the given row. 
	///-MatrixException:MatrixExceptions.ME_InexistantRow	
	_T RowSum(int row){
#ifndef MATH_NOERRORCHECK
		if(row>=rows){
			throw MathException(ME_InexistantRow);
		}
#endif
		_T ret=0;

		for(int col=0;col<cols;col++)
			ret+=data[col+row*cols];

		return ret;
	}
	
	////Returns sum of the given column. 
	///-MatrixException:MatrixExceptions.ME_InexistantColumn	
	_T ColSum(int col){
#ifndef MATH_NOERRORCHECK
		if(col>=cols){
			throw MathException(ME_InexistantColumn);
		}
#endif
		_T ret=0;

		for(int row=0;row<rows;row++)
			ret+=data[col+row*cols];

		return ret;
	}

	////Returns average of the given row.
	///-MatrixException:MatrixExceptions.ME_InexistantRow	
	_T RowAverage(int row) {
#ifndef MATH_NOERRORCHECK
		if(row>=rows){
			throw MathException(ME_InexistantRow);
		}
#endif
		return RowSum(row)/rows; 
	}

	////Returns average of the given column.
	///-MatrixException:MatrixExceptions.ME_InexistantColumn	
	_T ColAverage(int col) {
#ifndef MATH_NOERRORCHECK
		if(col>=cols){
			throw MathException(ME_InexistantColumn);
		}
#endif
		return ColSum(col)/cols;
	}

	////Swaps given two rows of the matrix.
	///-MatrixException:MatrixExceptions.ME_InexistantRow	
	basic_Matrix2D &SwapRows(int row1, int row2){
#ifndef MATH_NOERRORCHECK
		if(row1>=rows || row2>=rows){
			throw MathException(ME_InexistantRow);
		}
#endif
		int index1=row1*cols;
		int index2=row2*cols;
		for(int i=0;i<cols;i++){
			_T temp1=data[index1];
			_T temp2=data[index2];
			data[index1]=temp2;
			data[index2]=temp1;
			index1++;
			index2++;
		}

		return *this;

	}
	
	////Applies Gaussian Jordan Elimination to the matrix.
	///-MatrixException:MatrixExceptions.ME_MatrixEmpty

	basic_Matrix2D GaussianElimination(){
		
#ifndef MATH_NOERRORCHECK
		if(data.GetSize()==0 || (rows!=cols && rows!=cols-1)){
			throw MathException(ME_MatrixEmpty);
		}
#endif

		basic_Matrix2D temp=*this;


		for(int i=0;i<rows;i++) {
			if(temp(i,i)==0) {
				for(int row=0;row<rows;row++) {
					if(temp(i,row)!=0) {
						temp.SwapRows(i, row);
						break;
					}
				}
			}
		}

		for(int row=0;row<rows;row++){
			_T num=temp(row, row);

			for(int col=0;col<cols;col++){
				temp(row,col)/=num;		
			}

			for(int i=0;i<rows;i++){
				if(i!=row) {
					_T leading=temp(i, row);
					for(int col=0;col<cols;col++) {
						temp(i, col)-=temp(row, col)*leading;
					}
				}
			}			
			
		}

		temp.Temporary=true;
		return temp;
	}

	
	////Convert and returns all the data of matrix to a string.
	operator std::string() {
		int col,row;

		std::ostringstream str;

		for(col=0;col<rows;col++) {
			str<<"[ ";
			str.precision(3);
                        for(row=0;row<cols;row++) {
                            str<<std::setw(10)<<std::ios::fixed<<data[row+col*cols]<<" ";
                        }
			str<< " ]\n";
		}

		return str.str();
	}	


	//****************************************
	//***** SCALAR OPERATORS *****************
	//----- +=, -=, *=, /= operations --------
	//----- +, -, *, /, Sum operations -------

	////Sums each value, one by one, of the matrix with  
	/// the given constant number.
	basic_Matrix2D &operator +=(_T mult) {
		int i;
		for(i=0;i<rows*cols;i++)
			data[i]+=mult;

		return *this;
	}

	////Substracts the given constant number from the each  
	/// value, one by one, of matrix.
	basic_Matrix2D &operator -=(_T mult) {
		int i;
		for(i=0;i<rows*cols;i++)
			data[i]-=mult;

		return *this;
	}

	////Scalar multiplication:
	/// Multiplies the matrix by a given constant number.
	basic_Matrix2D &operator *=(_T mult) {
		int i;
		for(i=0;i<rows*cols;i++)
			data[i]*=mult;

		return *this;
	}
	////Scalar division:
	/// Divide the matrix with the given constant number.
	basic_Matrix2D &operator /=(_T mult) {
		int i;
		for(i=0;i<rows*cols;i++)
			data[i]/=mult;

		return *this;
	}
	////Sums each value, one by one, of the matrix with  
	/// the given constant number.
	/// This function also return the resultant matrix. 
	basic_Matrix2D operator +(_T mult) {
		basic_Matrix2D ret(GetSize());
		ret.Temporary=true;
		int i;
		for(i=0;i<rows*cols;i++)
			ret[i]=data[i]+mult;

		return ret;
	}

	////Substracts the given constant number from the each  
	/// value, one by one, of matrix.
	/// This funtion also returns the resultant matrix.
	basic_Matrix2D operator -(_T mult) {
		basic_Matrix2D ret(GetSize());
		ret.Temporary=true;
		int i;
		for(i=0;i<rows*cols;i++)
			ret[i]=data[i]-mult;

		return ret;
	}


	////Scalar multiplication:
	/// Multiplies the matrix by a given constant number.
	/// This function also returns the resultant matrix.
	basic_Matrix2D operator *(_T mult) {
		basic_Matrix2D ret(GetSize());
		ret.Temporary=true;
		int i;
		for(i=0;i<rows*cols;i++)
			ret[i]=data[i]*mult;

		return ret;
	}
	////Scalar division:
	/// Divide the matrix with the given constant number.
	/// This funtion also returns the resultant matrix.
	basic_Matrix2D operator /(_T mult) {
		basic_Matrix2D ret(GetSize());
		ret.Temporary=true;
		int i;
		for(i=0;i<rows*cols;i++)
			ret[i]=data[i]/mult;

		return ret;
	}


	//****************************************
	//***** MATRIX OPERATORS *****************
	//----- +=, -= operations ------------
	//----- +, -, * operations ------------
	//****************************************
	
	////Adds the given matrix to the current one.
	///	MatrixException:MatrixExceptions.ME_MatrixSizeNotEqual
	basic_Matrix2D &operator +=(basic_Matrix2D &mat1){
#ifndef  MATH_NOERRORCHECK
		if(mat1.rows!=rows || mat1.cols!=cols){
			throw MathException(ME_MatrixSizeNotEqual);
		}
#endif
		for(int col=0;col<rows;col++){
			for(int row=0;row<cols;row++){
				data[row+col*cols]+=mat1[row+col*cols];
			}
		}
		return *this;
	}

	////Substucts the given matrix from this matrix.
	///	MatrixException:MatrixExceptions.ME_MatrixSizeNotEqual
	basic_Matrix2D &operator -=(basic_Matrix2D &mat1){
#ifndef  MATH_NOERRORCHECK
		if(mat1.rows!=rows || mat1.cols!=cols){
			throw MathException(ME_MatrixSizeNotEqual);
		}
#endif
		for(int col=0;col<rows;col++){
			for(int row=0;row<cols;row++){
				data[row+col*cols]-=mat1[row+col*cols];
			}
		}
		return *this;
	}

	////Adds the given matrix to the current one.
	///	MatrixException:MatrixExceptions.ME_MatrixSizeNotEqual
	/// This function also returns the resultant matrix.
	basic_Matrix2D operator +(basic_Matrix2D &mat1){
#ifndef  MATH_NOERRORCHECK
		if(mat1.rows!=rows || mat1.cols!=cols){
			throw MathException(ME_MatrixSizeNotEqual);
		}
#endif
		
		if(cols!=mat1.cols || rows!=mat1.rows){

			throw MathException(ME_MatrixSizeNotEqual);
		
		}

		basic_Matrix2D temp(rows,cols);

		for(int row=0;row<rows;row++){
			for(int col=0;col<cols;col++){
				temp[col+row*cols]=data[col+row*cols]+mat1[col+row*cols];
			}
		}

		return temp;
	}
	////Substucts the given matrix from matrix.
	///	MatrixException:MatrixExceptions.ME_MatrixSizeNotEqual
	/// This function also returns the resultant matrix.
	basic_Matrix2D operator -(basic_Matrix2D &mat1){
#ifndef  MATH_NOERRORCHECK
		if(mat1.rows!=rows || mat1.cols!=cols){
			throw MathException(ME_MatrixSizeNotEqual);
		}
#endif		
		if(cols!=mat1.cols ||  rows!=mat1.rows){
			throw MathException(ME_MatrixSizeNotEqual);
		}
		basic_Matrix2D temp(rows,cols);

		for(int row=0;row<rows;row++){
			for(int col=0;col<cols;col++){
				temp[col+row*cols]=data[col+row*cols]-mat1[col+row*cols];
			}
		}

		return temp;
	}

	////Multiply the matrix by the given matrix.
	///	MatrixException:MatrixExceptions.ME_RowColumnMissMatch
	/// The size of resultant matrix is row(of first matrix)
	/// by column(of second matrix). 
	basic_Matrix2D operator *(basic_Matrix2D &mat1){
		
		if(cols!=mat1.rows){
		
			throw MathException(ME_RowColumnMissMatch);
			
		}
		basic_Matrix2D temp(rows,cols);

		for(int row=0;row<rows;row++){
			for(int col=0;col<cols;col++){
				temp[col+row*cols]=0;

				for(int k=0;k<cols;k++)
					temp[col+row*cols]+=data[k+row*cols]*mat1[col+k*cols];
			}
		}

		return temp;
	}



	////Find and returns the sum of all value of the matrix. 
	_T Sum() {
		_T res=0;
		int row,col;
		for(row=0;row<rows;row++){
			for(col=0;col<cols;col++)
				res+=data[col+row*cols];
	}

	return res;
	}

	
	//****************************************
	//***** SQUARE MATRICES ******************
	//----- Transpose, Minor, Cofactor -------
	//----- Determinant, Inverse, Identity ---
	//****************************************

	////Takes the ranspose of the matrix.
	///	MatrixException:MatrixExceptions.ME_MatrixNotSquare
	/// The function also returns the resultant matrix.
	basic_Matrix2D Transpose(){
#ifndef  MATH_NOERRORCHECK
		if(!isSquare()){
			throw MathException(ME_MatrixNotSquare);	
		}
#endif
		basic_Matrix2D ret(GetSize());
		ret.Temporary=true;
		int row,col;
		for(row=0;row<rows;row++){
			for(col=0;col<cols;col++)
				ret[col+row*cols]=data[row+col*rows];
		}
		
		return ret;
		
	}

	
	////Checks whether the matrix is identity matrix.
	///	MatrixException:MatrixExceptions.ME_MatrixNotSquare
	bool isIdentity(){
#ifndef  MATH_NOERRORCHECK
		if(!isSquare()){
			throw MathException(ME_MatrixNotSquare);
		}
#endif
		int row,col;
		_T v=std::numeric_limits<_T>::epsilon();
		if(v<1)
			v*=5;
		for(row=0;row<rows;row++){
			for(col=0;col<cols;col++){
				if(row==col){
					if(data[col+row*cols]-1>v || data[col+row*cols]-1<-v)
						return false;
					
				}else{
					if(data[col+row*cols]>v || data[col+row*cols]<-v)
						return false;					
				}
			}
		}
		return true;
	}

	
	
	////Takes the determinant of matrix.
	///	MatrixException:MatrixExceptions.ME_MatrixNotSquare
	_T Determinant(){
#ifndef  MATH_NOERRORCHECK
		if(!isSquare()){
			throw MathException(ME_MatrixNotSquare);
		}
#endif
		_T det=0;

		if(rows==2){
			det=data[0]*data[3]-data[1]*data[2];
		}
		
		else if(rows==3){

			_T temp;
			int col,k;

			for(col=0;col<cols;col++){
				temp=1;
				for(k=0;k<cols;k++)
					temp*=data[ k*cols + (col+k)%cols ];

				det+=temp;
			}



			for(col=0;col<cols;col++){
				temp=1;
				for(k=0;k<cols;k++)
					temp*=data[ k*cols + (col-k+cols)%cols ];

				det-=temp;
			}
		}
		else {
			basic_Matrix2D temp=FirstLineCofactor();
			for(int col=0;col<cols;col++)
				det+=data[col]*temp[col];
		}

		return det;

	}

	////Finds and returns inverse of the matrix.
	///	MatrixException:MatrixExceptions.ME_MatrixNotSquare
	/// The function also returns the resultant matrix.
	basic_Matrix2D Inverse(){
		
#ifndef MATH_NOERRORCHECK
		if(data.GetSize()==0 || (rows!=cols)){
			throw MathException(ME_MatrixNotSquare);
		}
#endif

		basic_Matrix2D temp;
		temp=*this;
		basic_Matrix2D ident=IdentityMatrix(rows);

		for(int i=0;i<rows;i++) {
			if(temp(i,i)==0) {
				for(int row=0;row<rows;row++) {
					if(temp(i,row)!=0) {
						temp.SwapRows(i, row);
						ident.SwapRows(i, row);
						break;
					}
				}
			}
		}

		for(int row=0;row<rows;row++){
			_T num=temp(row, row);

			for(int col=0;col<cols;col++){
				temp(row,col)/=num;	
				ident(row,col)/=num;
			}

			for(int i=0;i<rows;i++){
				if(i!=row) {
					_T leading=temp(i, row);
					for(int col=0;col<cols;col++) {
						temp(i, col)-=temp(row, col)*leading;
						ident(i, col)-=ident(row, col)*leading;
					}
				}
			}
			
			
		}

		temp.Temporary=true;
		ident.Temporary=true;
		return ident;
	}



	////Finds the Minor value of each elements of matrix.
	///	MatrixException:MatrixExceptions.ME_MatrixNotSquare
	basic_Matrix2D Minor() {
#ifndef  MATH_NOERRORCHECK
		if(!isSquare()){
			throw MathException(ME_MatrixNotSquare);
		}
#endif
		basic_Matrix2D temp,ret(GetSize());
		for(int row=0;row<rows;row++) {
			for(int col=0;col<cols;col++) {
				temp=*this;
				temp.RemoveRow(row);
				temp.RemoveCol(col);
				ret(row,col)=temp.Determinant();
			}
		}

		temp.Temporary=true;
		return ret;

	}

	////Finds the Cofactor value of each elements of matrix.
	///	MatrixException:MatrixExceptions.ME_MatrixNotSquare
	basic_Matrix2D Cofactor() {
#ifndef  MATH_NOERRORCHECK
		if(!isSquare()){
			throw MathException(ME_MatrixNotSquare);
		}
#endif
		basic_Matrix2D temp,ret(GetSize());
		_T s=1;
		for(int row=0;row<rows;row++) {
			for(int col=0;col<cols;col++) {
				temp=*this;
				ret(row,col)=temp.RemoveRow(row).RemoveCol(col).Determinant()*s;

				s*=-1;
			}

			if(cols%2==0) s*=-1;
		}

		temp.Temporary=true;
		return ret;
	}

 
	////Calculates adjoint of this matrix and returns it.
	///	MatrixException:MatrixExceptions.ME_MatrixNotSquare
	basic_Matrix2D Adjoint() {
#ifndef  MATH_NOERRORCHECK
		if(!isSquare()){
			throw MathException(ME_MatrixNotSquare);
		}
#endif
		return Cofactor().Transpose();
	}
	////Creates new identity matrix of given size.
	static basic_Matrix2D IdentityMatrix(int size){
		basic_Matrix2D mat(size,size);
		int row,col;
		for(row=0;row<size;row++)
			for(col=0;col<size;col++)
				mat.data[col+row*size]= (row==col ? 1 : 0);
		
		return mat;
	}
	////Creates a new Robert's cross filter matrix.
	static basic_Matrix2D RobertsCross() {
		basic_Matrix2D mat(2,2);
		mat(0,0)= 1;
		mat(1,0)=-1;
		mat(0,1)=-1;
		mat(1,1)= 1;

		return mat;
	}
	////Creates new Sobel filter matrix in X direction.
	static basic_Matrix2D SobelX() {
		basic_Matrix2D mat(3,3);
		mat(0,0)= 1;
		mat(1,0)= 2;
		mat(2,0)= 1;
		mat(0,1)= 0;
		mat(1,1)= 0;
		mat(2,1)= 0;
		mat(0,2)=-1;
		mat(1,2)=-2;
		mat(2,2)=-1;

		return mat;
	}
	////Creates new Sobel filter matrix in Y direction.
	static basic_Matrix2D SobelY() {
		basic_Matrix2D mat(3,3);
		mat(0,0)= 1;
		mat(0,1)= 2;
		mat(0,2)= 1;
		mat(1,0)= 0;
		mat(1,1)= 0;
		mat(1,2)= 0;
		mat(2,0)=-1;
		mat(2,1)=-2;
		mat(2,2)=-1;

		return mat;
	}
protected:
	//managed buffer
	ManagedBuffer<_T> &data;

	int rows;
	int cols;


	////Allows access of a matrix member addressing it
	/// by its index.
	_T &operator [](int index) {
		return data[index];
	}

private:

	//Finds Cofactor of first line
	basic_Matrix2D FirstLineCofactor() {

		basic_Matrix2D temp, wofirst,ret(1,cols);
		wofirst=*this;
		wofirst.RemoveRow(0);
		_T s=1;
		for(int col=0;col<cols;col++) {
			temp=wofirst;
			_T t=temp.RemoveCol(col).Determinant()*s;
			ret[col]=t;
			s*=-1;
		}
		

		ret.Temporary=true;
		return ret;
	}
};



////Allows streaming of matrix. It converts matrix to string,
/// every row is printed on a line enclosed in braces.
template <class _T>
std::ostream &operator << (std::ostream &out, basic_Matrix2D<_T> &matrix) {
	out<<(std::string)matrix<<std::endl;

	return out;
}


////Adds the textual form of the matrix to another string.
template <class _T>
std::string &operator + (std::string &out, basic_Matrix2D<_T> &matrix) {
        return out+(std::string)matrix;
}


////Standard double membered matrix.
typedef basic_Matrix2D<double> Matrix2D;
