#pragma once

#include <malloc.h>
#include <memory.h>
#include <set>

template <class _T>
class LinkedList;
template <class _T>
class IIterator;
template <class _T>
class LinkedListIterator;
template <class _T>
class LinkedListOrderedIterator;

////This class is a wrapper to all objects that will be added
/// to a linked list. This wrapper contains linked list structure,
/// order, the item and few other useful functions.
template <class _T>
class LinkedListItem {
	friend class LinkedList<_T>;
	friend class LinkedListIterator<_T>;
	friend class LinkedListOrderedIterator<_T>;
protected:
	LinkedListItem<_T> *prev,*next,*order_prev,*order_next;
	LinkedList<_T> *parent;
	float Order;
	
	LinkedListItem(LinkedList<_T> *parent) {
		prev=next=order_prev=order_next=NULL;
		this->parent=parent;
	}

public:
	////The item
	_T *Item;
	////Returns the next element in the list,
	/// if this is the last item returns NULL
	LinkedListItem<_T> *getNext() {
		return next;
	}
	////Returns the previous element in the list,
	/// if this is the first item returns NULL
	LinkedListItem<_T> *getPrevious() {
                return prev;
	}
	////Returns the next element in the ordered list,
	/// if this is the last item returns NULL
	LinkedListItem<_T> *getOrderedNext() {
		return order_next;
	}
	////Returns the previous element in the ordered list,
	/// if this is the first item returns NULL
	LinkedListItem<_T> *getOrderedPrevious() {
		return order_prev;
	}

	////Returns the order of this list item
	float getOrder() { return Order; }
	////Changes the order of this item
	void setOrder(float Order) {
		if(parent!=NULL)
			parent->setOrderOf(this,Order);
	}
};


////Generic iterator interface. 
template <class _T>
class IIterator {
public:
	////This method resets the iterator
	virtual void reset()=0;
	////This method gets the next item and moves item pointer to the next one
	virtual _T* get()=0;
	////This method returns whether iterator is at the last item
	virtual bool eof()=0;

	////This method gets the next item and moves item pointer to the next one
	virtual operator _T*() {
		return get();
	}
	////This method returns whether iterator is at the last item
	virtual operator bool() {
		return eof();
	}
	////This method resets the iterator
	virtual bool operator !() {
		return !eof();
	}
};


template <class _T>
////This iterator class iterates through unsorted linked list
class LinkedListIterator : public IIterator<_T> {
	friend class LinkedList<_T>;
protected:
	LinkedListItem<_T> *start,*current;
	bool reverse;

	LinkedListIterator(LinkedListItem<_T> *start,bool reverse) {
		current=start;
		this->start=start;
		this->reverse=reverse;
	}

public:
	////This method resets the iterator
	virtual void reset() {
		current=start;
	}

	////This method returns whether iterator is at the last item
	virtual bool eof() {
		return (current==NULL);
	}

	////This method gets the next item and moves item pointer to the next one
	virtual _T* get() {
		_T *item;

		if(current==NULL)
			return NULL;

		item=current->Item;

		if(reverse)
			current=current->prev;
		else
			current=current->next;

		return item;
	}

	////This method gets the next list item and moves item pointer to the next one
	LinkedListItem<_T>* getitem() {
		if(current==NULL)
			return NULL;

		LinkedListItem<_T>* item=current;

		if(reverse)
			current=current->prev;
		else
			current=current->next;

		return item;
	}
	////This method gets the next item and moves item pointer to the next one
	virtual operator LinkedListItem<_T>*() {
		return getitem();
	}
};
////This iterator class iterates through sorted linked list
template <class _T>
class LinkedListOrderedIterator : public IIterator<_T> {
	friend class LinkedList<_T>;
protected:
	LinkedListItem<_T> *start,*current;
	bool reverse;

	LinkedListOrderedIterator(LinkedListItem<_T>* start,bool reverse) {
		current=this->start=start;
		this->reverse=reverse;
	}

public:
	////This method resets the iterator
	virtual void reset() {
		current=start;
	}

	////This method returns whether iterator is at the last item
	virtual bool eof() {
		return (current==NULL);
	}

	////This method gets the next item and moves item pointer to the next one
	virtual _T* get() {
		_T *item;

		if(current==NULL)
			return NULL;

		item=current->Item;

		if(reverse)
			current=current->order_prev;
		else
			current=current->order_next;

		return item;
	}

	////This method gets the next list item and moves item pointer to the next one
	LinkedListItem<_T>* getitem() {
		if(current==NULL)
			return NULL;

		LinkedListItem<_T>* item=current;

		if(reverse)
			current=current->order_prev;
		else
			current=current->order_next;

		return item;
	}
	////This method gets the next item and moves item pointer to the next one
	virtual operator LinkedListItem<_T>*() {
		return getitem();
	}
};



////This class is a linked list structure.
/// It allows insert and remove operations
/// as well as sorted item query and iterators.
template <class _T>
class LinkedList {

	friend class LinkedListIterator<_T>;

protected:
	int poolid;
	int count;
	LinkedListItem<_T> **Pool;
	LinkedListItem<_T> *first,*last,*order_first,*order_last;

	void grow() {
		Pool=new LinkedListItem<_T>*[growth];
		int i;
		for(i=0;i<growth;i++)
			Pool[i]=new LinkedListItem<_T>(this);

		poolid=growth-1;
	}

	LinkedListItem<_T>* obtain() {
		if(poolid<0)
			grow();

		return Pool[poolid--];
	}
public:
	////The amount of growth for each step
	int growth;
	////Returns number of elements
	int getCount() {
		return count;
	}

	////Basic constructor
	LinkedList() {
		count=0;
		poolid=-1,
		growth=10;
		first=last=NULL;
		order_first=order_last=NULL;
	}

	////Changes the order of an item
	///@Position: position of the item to be reordered
	///@Order	: The new order
	void setOrderOf(int Position,float Order) {
		setOrderOf(ItemAt(Position),Order);
	}

	////Changes the order of an item
	///@Item	: The item to be reordered
	///@Order	: The new order
	void setOrderOf(LinkedListItem<_T> *Item,float Order) {
		if(Item==NULL)
			return;

		if(Item->Order==Order)
			return;

		if(Item->order_prev!=NULL)
			Item->order_prev->order_next=Item->order_next;

		if(Item->order_next!=NULL)
			Item->order_next->order_prev=Item->order_prev;

		if(order_first==Item)
			order_first=Item->order_next;

		if(order_last==Item)
			order_last=Item->order_prev;

		Item->Order=Order;

		bool done=false;

		LinkedListOrderedIterator<_T> i=GetOrderedIterator();
		LinkedListItem<_T> *item;
		LinkedListItem<_T> *ret=Item;
		while(item=i) {
			if(item->Order>=ret->Order) {
				if(item->order_prev!=NULL)
					item->order_prev->order_next=ret;
				else
					order_first=ret;

				ret->order_prev=item->order_prev;
				ret->order_next=item;

				item->order_prev=ret;

				done=true;

				break;
			}
		}

		if(!done) {
			if(order_last)
				order_last->order_next=ret;
			else
				order_first=ret;

			ret->order_prev=order_last;
			ret->order_next=NULL;
			order_last=ret;
		}

	}

	////Removes an item from the list
	void Remove(LinkedListItem<_T> *item) {
		if(item==NULL)
			return;

		if(item->prev!=NULL)
			item->prev->next=item->next;

		if(item->next!=NULL)
			item->next->prev=item->prev;

		if(first==item)
			first=item->next;

		if(last==item)
			last=item->prev;


		if(item->order_prev!=NULL)
			item->order_prev->order_next=item->order_next;

		if(item->order_next!=NULL)
			item->order_next->order_prev=item->order_prev;

		if(order_first==item)
			order_first=item->order_next;

		if(order_last==item)
			order_last=item->order_prev;

		count--;
		delete item;
	}

	////Gets the first item in list.
	/// Returns item itself
	_T *getFirstItem() {
		if(first)
			return first->Item;
		else
			return NULL;
	}

	////Gets the last item in list.
	/// Returns item itself
	_T *getLastItem() {
		if(last)
			return last->Item;
		else
			return NULL;
	}

	////Gets the first item in ordered list.
	/// Returns item itself
	_T *getOrderedFirstItem() {
		if(order_first)
			return order_first->Item;
		else
			return NULL;
	}

	////Gets the last item in ordered list.
	/// Returns item itself
	_T *getOrderedLastItem() {
		if(order_last)
			return order_last->Item;
		else
			return NULL;
	}

	////Gets the first item in list.
	/// Returns item wrapper
	LinkedListItem<_T> *getFirst() {
		return first;
	}

	////Gets the last item in list.
	/// Returns item wrapper
	LinkedListItem<_T> *getLast() {
		return last;
	}

	////Gets the first item in ordered list.
	/// Returns item wrapper
	LinkedListItem<_T> *getOrderedFirst() {
		return order_first;
	}

	////Gets the last item in ordered list.
	/// Returns item wrapper
	LinkedListItem<_T> *getOrderedLast() {
		return order_last;
	}

	float HighestOrder() {
		if(order_last)
			return order_last->Order;

		return 0;
	}

	float LowestOrder() {
		if(order_first)
			return order_first->Order;

		return 0;
	}

	////Removes an item from the list
	void Remove(_T *Item) {
		if(Item==NULL)
			return;
		
		LinkedListItem<_T>* item=FindListItem(Item);
		if(item==NULL)
			return;


		if(item->prev!=NULL)
			item->prev->next=item->next;

		if(item->next!=NULL)
			item->next->prev=item->prev;

		if(first==item)
			first=item->next;

		if(last==item)
			last=item->prev;


		if(item->order_prev!=NULL)
			item->order_prev->order_next=item->order_next;

		if(item->order_next!=NULL)
			item->order_next->order_prev=item->order_prev;

		if(order_first==item)
			order_first=item->order_next;

		if(order_last==item)
			order_last=item->order_prev;

		count--;
		delete item;
	}

	////Removes an item from the list
	void Remove(int Position) {
		Remove(ItemAt(Position));
	}

	////Clears the entire list
	void Clear() {
		while(getCount())
			Remove(0);
	}

	////Returns an item from a given position
	_T *operator [](int Position) {
		LinkedListItem<_T>* item=ListItemAt(Position);
		if(item)
			return item->Item;
		else
			return NULL;
	}

	////Returns an item from a given position
	_T *ItemAt(int Position) {
		LinkedListItem<_T>* item=ListItemAt(Position);
		if(item)
			return item->Item;
		else
			return NULL;
	}

	////Returns a list item from a given position
	LinkedListItem<_T> *ListItemAt(int Position) {
		LinkedListIterator<_T> i=(*this);
		LinkedListItem<_T> *item;

		while(item=i)
			if(Position--==0)
				return item;
	
		return NULL;
	}

	////Returns an item from a given ordered position
	_T *OrderedItemAt(int Position) {
		LinkedListItem<_T>* item=OrderedListItemAt(Position);
		if(item)
			return item->Item;
		else
			return NULL;
	}

	////Returns a list item from a given ordered position
	LinkedListItem<_T> *OrderedListItemAt(int Position) {
		LinkedListOrderedIterator<_T> i=(*this);
		LinkedListItem<_T> *item;

		while(item=i)
			if(Position--==0)
				return item;
	
		return NULL;
	}

	////Searches a specific item in the list and returns the list item.
	/// If item is not found NULL is returned
	LinkedListItem<_T> *FindListItem(_T *Item) {
		LinkedListIterator<_T> it=*this;
		LinkedListItem<_T> *item;
		while(item=it) {
			if(item->Item==Item)
				return item;
		}

		return NULL;
	}
	////Creates an ordered iterator object
	operator LinkedListOrderedIterator<_T>() {
		return GetOrderedIterator();
	}

	////Destroys the list by freeing every member and removing them
	void Destroy() {
		while(getCount()) {
			delete ItemAt(0);
			Remove(0);
		}
		int i;
		for(i=0;i<poolid;i++)
			delete Pool[i];
		delete Pool;
	}

	////Creates an iterator object
	operator LinkedListIterator<_T>() {
		return GetIterator();
	}

	////Creates an iterator object
	virtual LinkedListIterator<_T> GetIterator() {
		return LinkedListIterator<_T>(first,false);
	}

	////Creates a reverse iterator object
	virtual LinkedListIterator<_T> GetReverseIterator() {
		return LinkedListIterator<_T>(last,true);
	}

	////Creates an ordered iterator object
	virtual LinkedListOrderedIterator<_T> GetOrderedIterator() {
		return LinkedListOrderedIterator<_T>(order_first,false);
	}

	////Creates an ordered reverse iterator object
	virtual LinkedListOrderedIterator<_T> GetReverseOrderedIterator() {
		return LinkedListOrderedIterator<_T>(order_last,true);
	}

	LinkedListItem<_T> *AddItem(_T *Item) { return AddItem(Item, 0); }

	LinkedListItem<_T> *Add(_T *Item) { return AddItem(Item, 0); }

	////Adds a new item to the list. This function returns the list item
	/// that surrounds the newly added item
	///@Item	: The item to be added to the list
	///@Order	: The order of the item to be added
	LinkedListItem<_T> *AddItem(_T *Item,float Order) {
		LinkedListItem<_T> *ret=obtain();
		count++;

		ret->Order=Order;
		ret->Item=Item;

		if(first==NULL) {
			first=last=order_first=order_last=ret;
			ret->prev=NULL;
			ret->next=NULL;
			ret->order_prev=NULL;
			ret->order_next=NULL;
		}
		else {
			ret->prev=last;
			ret->next=NULL;

			last->next=ret;
			last=ret;

			LinkedListOrderedIterator<_T> i=GetOrderedIterator();
			LinkedListItem<_T> *item;

			bool done=false;

			while(item=i) {
				if(item->Order>=ret->Order) {
					if(item->order_prev!=NULL)
						item->order_prev->order_next=ret;
					else
						order_first=ret;

					ret->order_prev=item->order_prev;
					ret->order_next=item;

					item->order_prev=ret;

					done=true;

					break;
				}
			}

			if(!done) {
				if(order_last)
					order_last->order_next=ret;
				else
					order_first=ret;

				ret->order_prev=order_last;
				order_last=ret;
			}
		}

		return ret;
	}
};
